<?php
/***************************************
 * Configuraciones iniciales
 **************************************/
// Token con permisos de lectura/escritura en el repo
$githubToken = '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';

// Datos de tu repositorio (ajusta si hace falta)
$githubRepoOwner = 'luisoddone';
$githubRepoName  = 'femon';
$jsonFilePath    = 'peliseries.json';              // Nombre del archivo JSON en tu repo
$githubBranch    = 'main';

// Construimos la URL de la API de GitHub para tu archivo
$githubApiUrl = "https://api.github.com/repos/$githubRepoOwner/$githubRepoName/contents/$jsonFilePath?ref=$githubBranch";

// API Key de TMDb
$tmdbApiKey = 'eac6b66d86771fe225dfc28551bde098';

// Archivo-bandera para saber si ya se corrió la actualización masiva
$updateFlagFile = __DIR__ . '/.update_done';

/***************************************
 * Funciones de GitHub
 **************************************/
/***************************************
 * Funciones de GitHub
 **************************************/
/**
 * Obtiene el JSON desde GitHub, manejando archivos grandes (>1MB) de forma automática.
 *
 * @param string $apiUrl La URL completa a la API de contenidos de GitHub.
 * @param string $token Tu token de autenticación de GitHub.
 * @return array Un array con 'jsonArray' y 'sha'. Si falla, detiene el script.
 */
function getJsonFromGithub($apiUrl, $token) {
    // Para construir la URL del blob, necesitamos el owner y el repo,
    // que están definidos como variables globales en este script.
    global $githubRepoOwner, $githubRepoName;

    // --- PASO 1: Obtener la metadata del archivo (incluido el 'sha') ---
    $ch = curl_init($apiUrl);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        "Authorization: Bearer $token",
        "User-Agent: Mi-Panel-Peliculas"
    ]);
    $response = curl_exec($ch);
    $info = curl_getinfo($ch);
    curl_close($ch);

    if ($info['http_code'] !== 200) {
        die("Error (Paso 1) al obtener metadata del archivo. Código: {$info['http_code']}<br>Respuesta: <pre>" . htmlspecialchars($response) . "</pre>");
    }
    
    $data = json_decode($response, true);
    $fileSha = isset($data['sha']) ? $data['sha'] : null;

    if (!$fileSha) {
        die("Error: No se pudo obtener el 'sha' del archivo. Respuesta de GitHub: <pre>" . htmlspecialchars($response) . "</pre>");
    }

    $decodedContent = ''; // Inicializamos el contenido

    // --- PASO 2: Verificar si el contenido fue incluido o si el archivo es muy grande ---
    if (isset($data['content']) && !empty($data['content'])) {
        // Archivo pequeño, el contenido ya está en la primera respuesta.
        $decodedContent = base64_decode($data['content']);
    } elseif (isset($data['size']) && $data['size'] > 0) {
        // Archivo grande. El contenido viene vacío, así que lo obtenemos desde la API de Blobs.
        $blobUrl = "https://api.github.com/repos/$githubRepoOwner/$githubRepoName/git/blobs/$fileSha";
        
        $ch_blob = curl_init($blobUrl);
        curl_setopt($ch_blob, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch_blob, CURLOPT_HTTPHEADER, [
            "Authorization: Bearer $token",
            "User-Agent: Mi-Panel-Peliculas"
        ]);
        $blobResponse = curl_exec($ch_blob);
        $blobInfo = curl_getinfo($ch_blob);
        curl_close($ch_blob);

        if ($blobInfo['http_code'] !== 200) {
            die("Error (Paso 2) al obtener contenido del blob. Código: {$blobInfo['http_code']}<br>Respuesta: <pre>" . htmlspecialchars($blobResponse) . "</pre>");
        }

        $blobData = json_decode($blobResponse, true);
        if (isset($blobData['content'])) {
            $decodedContent = base64_decode($blobData['content']);
        } else {
            die("Error: La respuesta del blob no contenía el campo 'content'. Respuesta: <pre>" . htmlspecialchars($blobResponse) . "</pre>");
        }
    }
    // Si el archivo tiene un tamaño de 0, $decodedContent se queda como una cadena vacía, lo cual es correcto.

    // --- PASO 3: Decodificar el JSON y retornar el formato esperado ---
    $jsonArray = json_decode($decodedContent, true);
    
    // Nos aseguramos de que siempre sea un array, incluso si el archivo está vacío o mal formado.
    if (!is_array($jsonArray)) {
        $jsonArray = [];
    }

    return [
        'jsonArray' => $jsonArray,
        'sha'       => $fileSha
    ];
}

function updateJsonOnGithub($apiUrl, $token, $newContent, $sha, $mensajeCommit = "Actualizando JSON") {
    $payload = json_encode([
        "message" => $mensajeCommit,
        "content" => base64_encode($newContent),
        "sha"     => $sha
    ]);
    $ch = curl_init($apiUrl);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'PUT');
    curl_setopt($ch, CURLOPT_POSTFIELDS, $payload);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        "Authorization: Bearer $token",
        "Content-Type: application/json",
        "User-Agent: Mi-Panel-Peliculas"
    ]);
    curl_exec($ch);
    $info = curl_getinfo($ch);
    curl_close($ch);
    return in_array($info['http_code'], [200, 201]);
}

/***************************************
 * Funciones de TMDb
 **************************************/
function searchMovieTMDB($query, $tmdbApiKey) {
    $url = "https://api.themoviedb.org/3/search/movie?api_key=$tmdbApiKey&query=" . urlencode($query) . "&include_adult=false&language=es-ES";
    $data = json_decode(file_get_contents($url), true);
    return $data['results'] ?? [];
}

function getMovieDetails($movieId, $tmdbApiKey) {
    $url = "https://api.themoviedb.org/3/movie/$movieId?api_key=$tmdbApiKey&language=es-ES&append_to_response=images&include_image_language=es,null,en";
    return json_decode(file_get_contents($url), true);
}

function enrichMovieArray(&$movie, $tmdbApiKey) {
    // Si ya tiene descripción asumimos que está enriquecido
    if (!empty($movie['descripcion'] ?? '')) return;

    $searchResults = searchMovieTMDB($movie['name'], $tmdbApiKey);
    if (empty($searchResults)) return;

    // Seleccionamos el primer resultado
    $chosen = $searchResults[0];
    $details = getMovieDetails($chosen['id'], $tmdbApiKey);
    if (empty($details)) return;

    // Campos principales
    $movie['descripcion']      = $details['overview']             ?? '';
    $movie['anio']             = substr($details['release_date'] ?? '', 0, 4);
    $movie['duracion']         = ($details['runtime'] ?? 0) ? $details['runtime'] . ' min' : '';
    $movie['genero']           = !empty($details['genres']) ? implode(', ', array_column($details['genres'], 'name')) : '';

    // Backdrop / horizontal
    $movie['iconoHorizontal']  = !empty($details['backdrop_path']) ? 'https://image.tmdb.org/t/p/original' . $details['backdrop_path'] : '';

    // Logo PNG: buscamos primero español, luego inglés, luego cualquiera
    $iconPng = '';
    if (!empty($details['images']['logos'])) {
        $logos = $details['images']['logos'];
        $prioridad = ['es', 'en'];
        foreach ($prioridad as $lang) {
            foreach ($logos as $l) {
                if ($l['iso_639_1'] === $lang) {
                    $iconPng = 'https://image.tmdb.org/t/p/w500' . $l['file_path'];
                    break 2;
                }
            }
        }
        // Si no encontramos en los idiomas prioritarios
        if (!$iconPng) {
            $iconPng = 'https://image.tmdb.org/t/p/w500' . $logos[0]['file_path'];
        }
    }
    $movie['iconpng'] = $iconPng;
}

/***************************************
 * Cargar JSON inicial
 **************************************/
$githubData = getJsonFromGithub($githubApiUrl, $githubToken);
if (!$githubData) {
    die("Error al obtener el JSON de GitHub. Verifica token/URL.");
}
$jsonArray = $githubData['jsonArray'];
$currentSha = $githubData['sha'];

/***************************************
 * Variables de uso general
 **************************************/
$errorMsg      = "";
$successMsg    = "";
$searchResults = [];
$selectedMovie = null;

/***************************************
 * Acción: Actualización masiva (botón de un solo uso)
 **************************************/
if (isset($_GET['action']) && $_GET['action'] === 'runUpdate' && !file_exists($updateFlagFile)) {
    foreach ($jsonArray as &$cat) {
        foreach ($cat['samples'] as &$mov) {
            enrichMovieArray($mov, $tmdbApiKey);
        }
    }
    unset($mov, $cat); // limpiamos referencias

    $newJsonString = json_encode($jsonArray, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT);
    if (updateJsonOnGithub($githubApiUrl, $githubToken, $newJsonString, $currentSha, "Actualización masiva de películas")) {
        $successMsg = "Todas las películas fueron enriquecidas correctamente.";
        // Creamos el flag para que nunca más aparezca el botón
        file_put_contents($updateFlagFile, 'done');
        // Refrescamos el JSON/sha
        $upd = getJsonFromGithub($githubApiUrl, $githubToken);
        if ($upd) {
            $jsonArray  = $upd['jsonArray'];
            $currentSha = $upd['sha'];
        }
    } else {
        $errorMsg = "Ocurrió un problema al actualizar el JSON en GitHub.";
    }
}

/***************************************
 * Acción: Eliminar Película
 **************************************/
if (isset($_GET['action']) && $_GET['action'] === 'deleteMovie') {
    $cIndex = isset($_GET['cat']) ? (int)$_GET['cat'] : -1;
    $mIndex = isset($_GET['movie']) ? (int)$_GET['movie'] : -1;
    if ($cIndex >= 0 && isset($jsonArray[$cIndex]['samples'][$mIndex])) {
        $deletedMovie = $jsonArray[$cIndex]['samples'][$mIndex]['name'];
        array_splice($jsonArray[$cIndex]['samples'], $mIndex, 1);
        $newJsonString = json_encode($jsonArray, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT);
        if (updateJsonOnGithub($githubApiUrl, $githubToken, $newJsonString, $currentSha, "Eliminando película '$deletedMovie'")) {
            $successMsg = "Película '$deletedMovie' eliminada correctamente.";
            $updated = getJsonFromGithub($githubApiUrl, $githubToken);
            if ($updated) {
                $jsonArray  = $updated['jsonArray'];
                $currentSha = $updated['sha'];
            }
        } else {
            $errorMsg = "Error al eliminar la película.";
        }
    }
}

/***************************************
 * Nuevo: Mostrar formulario para mover película
 **************************************/
$moveMovieData = null;
if (isset($_GET['action']) && $_GET['action'] === 'showMoveMovieForm') {
    $cIndex = isset($_GET['cat']) ? (int)$_GET['cat'] : -1;
    $mIndex = isset($_GET['movie']) ? (int)$_GET['movie'] : -1;
    if ($cIndex >= 0 && isset($jsonArray[$cIndex]['samples'][$mIndex])) {
        $moveMovieData = [
            'catIndex'   => $cIndex,
            'movieIndex' => $mIndex,
            'name'       => $jsonArray[$cIndex]['samples'][$mIndex]['name']
        ];
    }
}

/***************************************
 * Nuevo: Acción mover película a otra categoría
 **************************************/
if (isset($_GET['action']) && $_GET['action'] === 'moveToCategory') {
    $fromCat    = isset($_GET['cat'])       ? (int)$_GET['cat']       : -1;
    $movieIndex = isset($_GET['movie'])     ? (int)$_GET['movie']     : -1;
    $toCat      = isset($_GET['targetCat']) ? (int)$_GET['targetCat'] : -1;

    if ($fromCat >= 0 && $toCat >= 0
        && isset($jsonArray[$fromCat]['samples'][$movieIndex])
        && isset($jsonArray[$toCat])) {

        $movieObj = $jsonArray[$fromCat]['samples'][$movieIndex];
        array_splice($jsonArray[$fromCat]['samples'], $movieIndex, 1);
        $jsonArray[$toCat]['samples'][] = $movieObj;

        $newJsonString = json_encode($jsonArray, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT);
        if (updateJsonOnGithub($githubApiUrl, $githubToken, $newJsonString, $currentSha, "Moviendo película a otra categoría")) {
            $successMsg = "Película '{$movieObj['name']}' movida a categoría '{$jsonArray[$toCat]['name']}' correctamente.";
            $updated = getJsonFromGithub($githubApiUrl, $githubToken);
            if ($updated) {
                $jsonArray  = $updated['jsonArray'];
                $currentSha = $updated['sha'];
            }
        } else {
            $errorMsg = "Error al mover la película a otra categoría.";
        }
    }
}

/***************************************
 * Acciones para mover categorías
 **************************************/
if (isset($_GET['action']) && $_GET['action'] === 'moveCategory') {
    $index = isset($_GET['index']) ? (int)$_GET['index'] : -1;
    $dir   = isset($_GET['dir'])   ? $_GET['dir']         : '';

    if ($index >= 0 && $index < count($jsonArray)) {
        if ($dir === 'up' && $index > 0) {
            $temp = $jsonArray[$index];
            $jsonArray[$index] = $jsonArray[$index - 1];
            $jsonArray[$index - 1] = $temp;
        } elseif ($dir === 'down' && $index < count($jsonArray) - 1) {
            $temp = $jsonArray[$index];
            $jsonArray[$index] = $jsonArray[$index + 1];
            $jsonArray[$index + 1] = $temp;
        }
        $newJsonString = json_encode($jsonArray, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT);
        if (updateJsonOnGithub($githubApiUrl, $githubToken, $newJsonString, $currentSha, "Moviendo categoría")) {
            $successMsg = "Categoría movida con éxito.";
            $updated = getJsonFromGithub($githubApiUrl, $githubToken);
            if ($updated) {
                $jsonArray  = $updated['jsonArray'];
                $currentSha = $updated['sha'];
            }
        } else {
            $errorMsg = "No se pudo actualizar el JSON al mover la categoría.";
        }
    }
}

/***************************************
 * Acciones para mover películas dentro de su categoría
 **************************************/
if (isset($_GET['action']) && $_GET['action'] === 'moveMovie') {
    $catIndex   = isset($_GET['cat'])   ? (int)$_GET['cat']   : -1;
    $movieIndex = isset($_GET['movie']) ? (int)$_GET['movie'] : -1;
    $dir        = isset($_GET['dir'])   ? $_GET['dir']        : '';

    if ($catIndex >= 0 && $catIndex < count($jsonArray)) {
        $samples = &$jsonArray[$catIndex]['samples'];
        if ($movieIndex >= 0 && $movieIndex < count($samples)) {
            if ($dir === 'up' && $movieIndex > 0) {
                $temp = $samples[$movieIndex];
                $samples[$movieIndex] = $samples[$movieIndex - 1];
                $samples[$movieIndex - 1] = $temp;
            } elseif ($dir === 'down' && $movieIndex < count($samples) - 1) {
                $temp = $samples[$movieIndex];
                $samples[$movieIndex] = $samples[$movieIndex + 1];
                $samples[$movieIndex + 1] = $temp;
            }
            $newJsonString = json_encode($jsonArray, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT);
            if (updateJsonOnGithub($githubApiUrl, $githubToken, $newJsonString, $currentSha, "Moviendo película")) {
                $successMsg = "Película movida con éxito.";
                $updated = getJsonFromGithub($githubApiUrl, $githubToken);
                if ($updated) {
                    $jsonArray  = $updated['jsonArray'];
                    $currentSha = $updated['sha'];
                }
            } else {
                $errorMsg = "No se pudo actualizar el JSON al mover la película.";
            }
        }
    }
}

/***************************************
 * Acción: Buscar Películas en TMDb
 **************************************/
if (isset($_GET['action']) && $_GET['action'] === 'search' && !empty($_GET['query'])) {
    $searchResults = searchMovieTMDB(trim($_GET['query']), $tmdbApiKey);
}

/***************************************
 * Acción: Seleccionar Película de la Búsqueda en TMDb
 **************************************/
if (isset($_GET['action']) && $_GET['action'] === 'select' && !empty($_GET['id'])) {
    $tmdbId      = (int)$_GET['id'];
    $searchTitle = isset($_GET['st']) ? trim($_GET['st']) : '';
    $details     = getMovieDetails($tmdbId, $tmdbApiKey);
    if (empty($details['title'])) {
        $errorMsg = "No se encontró la película en TMDb o falta el título.";
    } else {
        // Construimos estructura completa para pre-cargar en el form
        $poster = !empty($details['poster_path'])
            ? 'https://image.tmdb.org/t/p/original' . $details['poster_path']
            : '';
        $logoPng = '';
        if (!empty($details['images']['logos'])) {
            $logos = $details['images']['logos'];
            $prioridad = ['es', 'en'];
            foreach ($prioridad as $lang) {
                foreach ($logos as $l) {
                    if ($l['iso_639_1'] === $lang) {
                        $logoPng = 'https://image.tmdb.org/t/p/w500' . $l['file_path'];
                        break 2;
                    }
                }
            }
            if (!$logoPng) $logoPng = 'https://image.tmdb.org/t/p/w500' . $logos[0]['file_path'];
        }
        $selectedMovie = [
            'name'            => $details['title'] ?: $searchTitle,
            'icono'           => $poster,
            'iconoHorizontal' => !empty($details['backdrop_path']) ? 'https://image.tmdb.org/t/p/original' . $details['backdrop_path'] : '',
            'iconpng'         => $logoPng,
            'descripcion'     => $details['overview']            ?? '',
            'anio'            => substr($details['release_date'] ?? '', 0, 4),
            'genero'          => !empty($details['genres']) ? implode(', ', array_column($details['genres'], 'name')) : '',
            'duracion'        => ($details['runtime'] ?? 0) ? $details['runtime'] . ' min' : ''
        ];
    }
}

/***************************************
 * Acción: Guardar Nueva Película
 **************************************/
if (isset($_POST['action']) && $_POST['action'] === 'save') {
    $newName        = trim($_POST['name']);
    $newIcono       = trim($_POST['icono']);
    $newUrl         = trim($_POST['url']);
    $newIconoHor    = trim($_POST['iconoHorizontal'] ?? '');
    $newIconpng     = trim($_POST['iconpng'] ?? '');
    $newDesc        = trim($_POST['descripcion'] ?? '');
    $newAnio        = trim($_POST['anio'] ?? '');
    $newGenero      = trim($_POST['genero'] ?? '');
    $newDuracion    = trim($_POST['duracion'] ?? '');
    $catSelect      = trim($_POST['categoria']);
    $catNew         = trim($_POST['categoria_nueva']);

    $exists = false;
    foreach ($jsonArray as $cat) {
        foreach ($cat['samples'] as $pel) {
            if (strcasecmp($pel['name'], $newName) === 0) {
                $exists = true;
                break 2;
            }
        }
    }
    if ($exists) {
        $errorMsg = "La película '$newName' ya existe en el JSON.";
    } else {
        $nuevo = [
            "name"            => $newName,
            "url"             => $newUrl,
            "icono"           => $newIcono,
            "iconoHorizontal" => $newIconoHor,
            "iconpng"         => $newIconpng,
            "descripcion"     => $newDesc,
            "anio"            => $newAnio,
            "genero"          => $newGenero,
            "duracion"        => $newDuracion,
            "type"            => "PELICULA"
        ];
        if ($catSelect === 'nueva' && $catNew !== '') {
            $jsonArray[] = ["name" => $catNew, "samples" => [$nuevo]];
        } else {
            foreach ($jsonArray as $i => $cat) {
                if ($cat['name'] === $catSelect) {
                    $jsonArray[$i]['samples'][] = $nuevo;
                    break;
                }
            }
        }
        if (!$errorMsg) {
            $jsonStr = json_encode($jsonArray, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
            if (updateJsonOnGithub($githubApiUrl, $githubToken, $jsonStr, $currentSha, "Agregando película '$newName'")) {
                $successMsg = "¡Película '$newName' agregada correctamente!";
                $upd = getJsonFromGithub($githubApiUrl, $githubToken);
                if ($upd) {
                    $jsonArray  = $upd['jsonArray'];
                    $currentSha = $upd['sha'];
                }
            } else {
                $errorMsg = "Error al actualizar el JSON en GitHub. Verifica permisos.";
            }
        }
    }
}

/***************************************
 * Acción: Editar Película Existente
 **************************************/
$editMovieData = null;
if (isset($_GET['action']) && $_GET['action'] === 'editMovie') {
    $c = isset($_GET['cat'])   ? (int)$_GET['cat']   : -1;
    $m = isset($_GET['movie']) ? (int)$_GET['movie'] : -1;
    if ($c >= 0 && isset($jsonArray[$c]['samples'][$m])) {
        $pel = $jsonArray[$c]['samples'][$m];
        $editMovieData = [
            'catIndex'        => $c,
            'movieIndex'      => $m,
            'name'            => $pel['name'],
            'url'             => $pel['url'],
            'icono'           => $pel['icono'],
            'iconoHorizontal' => $pel['iconoHorizontal'] ?? '',
            'iconpng'         => $pel['iconpng']         ?? '',
            'descripcion'     => $pel['descripcion']     ?? '',
            'anio'            => $pel['anio']            ?? '',
            'genero'          => $pel['genero']          ?? '',
            'duracion'        => $pel['duracion']        ?? ''
        ];
    }
}

/***************************************
 * Acción: Actualizar Película (después de editar)
 **************************************/
if (isset($_POST['action']) && $_POST['action'] === 'updateMovie') {
    $cIndex        = (int)$_POST['catIndex'];
    $mIndex        = (int)$_POST['movieIndex'];
    $newName       = trim($_POST['name']);
    $newUrl        = trim($_POST['url']);
    $newIcono      = trim($_POST['icono']);
    $newIconoHor   = trim($_POST['iconoHorizontal']);
    $newIconpng    = trim($_POST['iconpng']);
    $newDesc       = trim($_POST['descripcion']);
    $newAnio       = trim($_POST['anio']);
    $newGenero     = trim($_POST['genero']);
    $newDuracion   = trim($_POST['duracion']);
    $newPos        = (int)$_POST['position'];

    if (isset($jsonArray[$cIndex]['samples'][$mIndex])) {
        $jsonArray[$cIndex]['samples'][$mIndex]['name']            = $newName;
        $jsonArray[$cIndex]['samples'][$mIndex]['url']             = $newUrl;
        $jsonArray[$cIndex]['samples'][$mIndex]['icono']           = $newIcono;
        $jsonArray[$cIndex]['samples'][$mIndex]['iconoHorizontal'] = $newIconoHor;
        $jsonArray[$cIndex]['samples'][$mIndex]['iconpng']         = $newIconpng;
        $jsonArray[$cIndex]['samples'][$mIndex]['descripcion']     = $newDesc;
        $jsonArray[$cIndex]['samples'][$mIndex]['anio']            = $newAnio;
        $jsonArray[$cIndex]['samples'][$mIndex]['genero']          = $newGenero;
        $jsonArray[$cIndex]['samples'][$mIndex]['duracion']        = $newDuracion;

        $samples    = $jsonArray[$cIndex]['samples'];
        $target     = $newPos - 1;
        if ($target >= 0 && $target < count($samples) && $target !== $mIndex) {
            $pelData = $samples[$mIndex];
            array_splice($samples, $mIndex, 1);
            array_splice($samples, $target,    0, [$pelData]);
            $jsonArray[$cIndex]['samples'] = $samples;
        }

        $jsonStr = json_encode($jsonArray, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
        if (updateJsonOnGithub($githubApiUrl, $githubToken, $jsonStr, $currentSha, "Editando película '$newName'")) {
            $successMsg = "Película '$newName' editada exitosamente.";
            $upd = getJsonFromGithub($githubApiUrl, $githubToken);
            if ($upd) {
                $jsonArray  = $upd['jsonArray'];
                $currentSha = $upd['sha'];
            }
        } else {
            $errorMsg = "Error al actualizar el JSON al editar película.";
        }
    }
}

/***************************************
 * Acción: Actualizar Posición de Categoría
 **************************************/
if (isset($_POST['action']) && $_POST['action'] === 'updateCategory') {
    $catIndex = (int)$_POST['catIndex'];
    $newPos   = (int)$_POST['position'];
    if (isset($jsonArray[$catIndex])) {
        $cnt       = count($jsonArray);
        $target    = $newPos - 1;
        if ($target >= 0 && $target < $cnt && $target !== $catIndex) {
            $catData = $jsonArray[$catIndex];
            array_splice($jsonArray, $catIndex, 1);
            array_splice($jsonArray, $target,    0, [$catData]);
            $jsonStr = json_encode($jsonArray, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
            if (updateJsonOnGithub($githubApiUrl, $githubToken, $jsonStr, $currentSha, "Reordenando categorías")) {
                $successMsg = "Categoría movida exitosamente.";
                $upd = getJsonFromGithub($githubApiUrl, $githubToken);
                if ($upd) {
                    $jsonArray  = $upd['jsonArray'];
                    $currentSha = $upd['sha'];
                }
            } else {
                $errorMsg = "Error al actualizar el JSON al reordenar categorías.";
            }
        } else {
            $errorMsg = "Posición inválida.";
        }
    } else {
        $errorMsg = "Categoría no encontrada.";
    }
}

/***************************************
 * Acción: Actualizar Nombre de Categoría
 **************************************/
if (isset($_POST['action']) && $_POST['action'] === 'updateCategoryName') {
    $catIndex = (int)$_POST['catIndex'];
    $newName  = trim($_POST['newName']);
    if (isset($jsonArray[$catIndex])) {
        $jsonArray[$catIndex]['name'] = $newName;
        $jsonStr = json_encode($jsonArray, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
        if (updateJsonOnGithub($githubApiUrl, $githubToken, $jsonStr, $currentSha, "Actualizando nombre de categoría")) {
            $successMsg = "Nombre de categoría actualizado.";
            $upd = getJsonFromGithub($githubApiUrl, $githubToken);
            if ($upd) {
                $jsonArray  = $upd['jsonArray'];
                $currentSha = $upd['sha'];
            }
        } else {
            $errorMsg = "Error al actualizar nombre de categoría.";
        }
    } else {
        $errorMsg = "Categoría no encontrada.";
    }
}

/***************************************
 * Acción: Buscar Películas Existentes en JSON
 **************************************/
$localSearchResults = [];
if (isset($_GET['action']) && $_GET['action'] === 'searchExisting' && !empty($_GET['query'])) {
    $q = trim($_GET['query']);
    foreach ($jsonArray as $ci => $cat) {
        foreach ($cat['samples'] as $mi => $pel) {
            if (stripos($pel['name'], $q) !== false) {
                $localSearchResults[] = [
                    'catIndex'   => $ci,
                    'movieIndex' => $mi,
                    'name'       => $pel['name'],
                    'url'        => $pel['url'],
                    'icono'      => $pel['icono'],
                    'catName'    => $cat['name']
                ];
            }
        }
    }
}

/***************************************
 * Acciones para editar categoría
 **************************************/
$editCategoryNameData     = null;
$editCategoryPositionData = null;
if (isset($_GET['action']) && $_GET['action'] === 'editCategoryName') {
    $ci = isset($_GET['cat']) ? (int)$_GET['cat'] : -1;
    if ($ci >= 0 && isset($jsonArray[$ci])) {
        $editCategoryNameData = ['catIndex'=>$ci,'name'=>$jsonArray[$ci]['name']];
    }
} elseif (isset($_GET['action']) && $_GET['action'] === 'editCategoryPosition') {
    $ci = isset($_GET['cat']) ? (int)$_GET['cat'] : -1;
    if ($ci >= 0 && isset($jsonArray[$ci])) {
        $editCategoryPositionData = ['catIndex'=>$ci,'name'=>$jsonArray[$ci]['name']];
    }
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <title>Panel Admin - Películas</title>
    <style>
        body { background:#f0f2f5; margin:0; padding:0; font-family:Arial,sans-serif; }
        .container { max-width:1100px; margin:30px auto; background:#fff; padding:25px; border-radius:8px; box-shadow:0 0 10px rgba(0,0,0,0.1); }
        h1,h2 { margin:0 0 15px; font-weight:normal; }
        .error { color:#d9534f; font-weight:bold; margin-bottom:15px; }
        .success { color:#5cb85c; font-weight:bold; margin-bottom:15px; }
        .block { margin-bottom:30px; border-bottom:1px solid #dedede; padding-bottom:20px; }
        .block h2 { font-size:1.3em; margin-bottom:10px; border-left:4px solid #007bff; padding-left:8px; }
        .input-group { margin-bottom:12px; }
        .label { display:inline-block; width:140px; font-weight:bold; }
        input[type="text"], textarea, select { width:70%; padding:8px; box-sizing:border-box; font-size:1em; margin-left:5px; border:1px solid #ccc; border-radius:4px; }
        textarea { resize:vertical; }
        button { padding:10px 20px; background:#007bff; color:#fff; border:none; cursor:pointer; border-radius:4px; font-size:1em; }
        button:hover { background:#0056b3; }
        .edit-link, .button-like { display:inline-block; margin-left:8px; padding:6px 10px; background:#28a745; color:#fff!important; text-decoration:none; border-radius:4px; font-size:0.9em; }
        .edit-link:hover, .button-like:hover { background:#218838; }
        .delete-link { display:inline-block; margin-left:8px; padding:6px 10px; background:#d9534f; color:#fff!important; text-decoration:none; border-radius:4px; font-size:0.9em; }
        .delete-link:hover { background:#c9302c; }
        .search-results ul, .category-list ul { list-style:none; padding-left:0; }
        .search-results li, .category-list li { margin:5px 0; }
        .move-links { margin-left:10px; font-size:0.9em; }
        .move-links a { color:#666; text-decoration:none; margin-right:4px; }
        .move-links a:hover { text-decoration:underline; }
        .divider { border-top:1px dashed #ccc; margin:15px 0; }
        .pos-field { width:60px!important; }
        .update-btn { background:#ff6600; }
        .update-btn:hover { background:#cc5200; }
    </style>
</head>
<body>
<div class="container">
    <h1>Panel de Administración de Películas
        <?php if (!file_exists($updateFlagFile)): ?>
            <!-- Botón de un solo uso -->
            <a href="?action=runUpdate" class="button-like update-btn" onclick="return confirm('Esta acción actualizará TODAS las películas y el botón desaparecerá para siempre. ¿Continuar?');">Actualizar</a>
        <?php endif; ?>
    </h1>

    <?php if($errorMsg): ?>
        <div class="error"><?= htmlspecialchars($errorMsg) ?></div>
    <?php endif; ?>
    <?php if($successMsg): ?>
        <div class="success"><?= htmlspecialchars($successMsg) ?></div>
    <?php endif; ?>

    <!-- Bloque 1: Buscar Película en TMDb -->
    <div class="block">
        <h2>1) Buscar Película en TMDb</h2>
        <form method="get" action="">
            <input type="hidden" name="action" value="search" />
            <div class="input-group">
                <label for="query" class="label">Título:</label>
                <input type="text" name="query" id="query" placeholder="Ej: Titanic" required />
            </div>
            <button type="submit">Buscar</button>
        </form>
        <?php if (!empty($searchResults)): ?>
            <div class="search-results">
                <h3>Resultados de búsqueda:</h3>
                <ul>
                <?php foreach ($searchResults as $r): ?>
                    <?php
                        $title  = $r['title'] ?? '';
                        $year   = substr($r['release_date'] ?? '', 0,4);
                        $tmdbId = $r['id'];
                    ?>
                    <li>
                        <?= htmlspecialchars($title) ?> (<?= htmlspecialchars($year) ?>)
                        <a href="?action=select&id=<?= $tmdbId ?>&st=<?= urlencode($title) ?>">[Seleccionar]</a>
                    </li>
                <?php endforeach; ?>
                </ul>
            </div>
        <?php endif; ?>
    </div>

    <!-- Bloque 2: Agregar Película Seleccionada -->
    <div class="block">
        <h2>2) Agregar Película Seleccionada</h2>
        <?php if ($selectedMovie): ?>
            <form method="post" action="">
                <input type="hidden" name="action" value="save" />
                <div class="input-group">
                    <label class="label" for="name">Nombre:</label>
                    <input type="text" name="name" id="name" value="<?= htmlspecialchars($selectedMovie['name']) ?>" readonly />
                </div>
                <div class="input-group">
                    <label class="label" for="icono">Póster:</label>
                    <input type="text" name="icono" id="icono" value="<?= htmlspecialchars($selectedMovie['icono']) ?>" readonly />
                </div>
                <div class="input-group">
                    <label class="label" for="iconoHorizontal">Backdrop:</label>
                    <input type="text" name="iconoHorizontal" id="iconoHorizontal" value="<?= htmlspecialchars($selectedMovie['iconoHorizontal']) ?>" readonly />
                </div>
                <div class="input-group">
                    <label class="label" for="iconpng">Logo PNG:</label>
                    <input type="text" name="iconpng" id="iconpng" value="<?= htmlspecialchars($selectedMovie['iconpng']) ?>" readonly />
                </div>
                <div class="input-group">
                    <label class="label" for="descripcion">Sinopsis:</label>
                    <textarea name="descripcion" id="descripcion" rows="3" readonly><?= htmlspecialchars($selectedMovie['descripcion']) ?></textarea>
                </div>
                <div class="input-group">
                    <label class="label" for="anio">Año:</label>
                    <input type="text" name="anio" id="anio" value="<?= htmlspecialchars($selectedMovie['anio']) ?>" readonly />
                </div>
                <div class="input-group">
                    <label class="label" for="genero">Género:</label>
                    <input type="text" name="genero" id="genero" value="<?= htmlspecialchars($selectedMovie['genero']) ?>" readonly />
                </div>
                <div class="input-group">
                    <label class="label" for="duracion">Duración:</label>
                    <input type="text" name="duracion" id="duracion" value="<?= htmlspecialchars($selectedMovie['duracion']) ?>" readonly />
                </div>
                <div class="input-group">
                    <label class="label" for="url">URL de la Película:</label>
                    <input type="text" name="url" id="url" placeholder="Enlace .mp4 o streaming" required />
                </div>
                <div class="divider"></div>
                <h3>Categoría</h3>
                <div class="input-group">
                    <label class="label" for="categoria">Elegir:</label>
                    <select name="categoria" id="categoria">
                        <?php foreach($jsonArray as $cat): ?>
                            <option value="<?= htmlspecialchars($cat['name']) ?>"><?= htmlspecialchars($cat['name']) ?></option>
                        <?php endforeach; ?>
                        <option value="nueva">-- Nueva categoría --</option>
                    </select>
                </div>
                <div class="input-group">
                    <label class="label" for="categoria_nueva">Nueva cat.:</label>
                    <input type="text" name="categoria_nueva" id="categoria_nueva" placeholder="Solo si elegiste 'nueva'" />
                </div>
                <button type="submit">Guardar en JSON</button>
            </form>
        <?php else: ?>
            <p><em>No hay película seleccionada. Realiza una búsqueda y haz clic en <strong>“Seleccionar”</strong>.</em></p>
        <?php endif; ?>
    </div>

    <!-- Bloque: Buscador de Películas Existentes -->
    <div class="block">
        <h2>Buscador de Películas Existentes</h2>
        <form method="get" action="">
            <input type="hidden" name="action" value="searchExisting" />
            <div class="input-group">
                <label for="query" class="label">Buscar en JSON:</label>
                <input type="text" name="query" id="query" placeholder="Ej: Titanic" required />
            </div>
            <button type="submit">Buscar</button>
        </form>
        <?php if (!empty($localSearchResults)): ?>
            <div class="search-results">
                <h3>Resultados de búsqueda en JSON:</h3>
                <ul>
                    <?php foreach($localSearchResults as $res): ?>
                        <li>
                            <?= htmlspecialchars($res['name']) ?> (Categoría: <?= htmlspecialchars($res['catName']) ?>)
                            <a class="edit-link" href="?action=editMovie&cat=<?= $res['catIndex'] ?>&movie=<?= $res['movieIndex'] ?>">Editar</a>
                            <span class="move-links">
                                <a href="?action=moveMovie&cat=<?= $res['catIndex'] ?>&movie=<?= $res['movieIndex'] ?>&dir=up">[↑]</a>
                                <a href="?action=moveMovie&cat=<?= $res['catIndex'] ?>&movie=<?= $res['movieIndex'] ?>&dir=down">[↓]</a>
                            </span>
                        </li>
                    <?php endforeach; ?>
                </ul>
            </div>
        <?php elseif (isset($_GET['action']) && $_GET['action']==='searchExisting'): ?>
            <p><em>No se encontraron películas que coincidan con tu búsqueda.</em></p>
        <?php endif; ?>
    </div>

    <!-- Bloque: Editar Película Existente -->
    <?php if ($editMovieData): ?>
    <div class="block">
        <h2>Editar Película: <?= htmlspecialchars($editMovieData['name']) ?></h2>
        <form method="post" action="">
            <input type="hidden" name="action" value="updateMovie" />
            <input type="hidden" name="catIndex" value="<?= $editMovieData['catIndex'] ?>" />
            <input type="hidden" name="movieIndex" value="<?= $editMovieData['movieIndex'] ?>" />
            <div class="input-group">
                <label class="label" for="name">Nombre:</label>
                <input type="text" name="name" id="name" value="<?= htmlspecialchars($editMovieData['name']) ?>" />
            </div>
            <div class="input-group">
                <label class="label" for="url">URL:</label>
                <input type="text" name="url" id="url" value="<?= htmlspecialchars($editMovieData['url']) ?>" />
            </div>
            <div class="input-group">
                <label class="label" for="icono">Póster:</label>
                <input type="text" name="icono" id="icono" value="<?= htmlspecialchars($editMovieData['icono']) ?>" />
            </div>
            <div class="input-group">
                <label class="label" for="iconoHorizontal">Backdrop:</label>
                <input type="text" name="iconoHorizontal" id="iconoHorizontal" value="<?= htmlspecialchars($editMovieData['iconoHorizontal']) ?>" />
            </div>
            <div class="input-group">
                <label class="label" for="iconpng">Logo PNG:</label>
                <input type="text" name="iconpng" id="iconpng" value="<?= htmlspecialchars($editMovieData['iconpng']) ?>" />
            </div>
            <div class="input-group">
                <label class="label" for="descripcion">Sinopsis:</label>
                <textarea name="descripcion" id="descripcion" rows="3"><?= htmlspecialchars($editMovieData['descripcion']) ?></textarea>
            </div>
            <div class="input-group">
                <label class="label" for="anio">Año:</label>
                <input type="text" name="anio" id="anio" value="<?= htmlspecialchars($editMovieData['anio']) ?>" />
            </div>
            <div class="input-group">
                <label class="label" for="genero">Género:</label>
                <input type="text" name="genero" id="genero" value="<?= htmlspecialchars($editMovieData['genero']) ?>" />
            </div>
            <div class="input-group">
                <label class="label" for="duracion">Duración:</label>
                <input type="text" name="duracion" id="duracion" value="<?= htmlspecialchars($editMovieData['duracion']) ?>" />
            </div>
            <div class="divider"></div>
            <h3>Mover dentro de la misma categoría</h3>
            <?php
                $cI = $editMovieData['catIndex'];
                $mI = $editMovieData['movieIndex'];
                $cnt = count($jsonArray[$cI]['samples']);
            ?>
            <div class="input-group">
                <label class="label" for="position">Posición:</label>
                <select name="position" id="position" class="pos-field">
                    <?php for ($i=1; $i<=$cnt; $i++): ?>
                        <option value="<?= $i ?>" <?= $i==($mI+1)?'selected':'' ?>><?= $i ?></option>
                    <?php endfor; ?>
                </select>
            </div>
            <button type="submit">Guardar Cambios</button>
        </form>
    </div>
    <?php endif; ?>

    <!-- Bloque inline para mover entre categorías -->
    <?php if ($moveMovieData): ?>
    <div class="block">
        <h2>Agregar «<?= htmlspecialchars($moveMovieData['name']) ?>» a otra categoría</h2>
        <form method="get" action="">
            <input type="hidden" name="action" value="moveToCategory" />
            <input type="hidden" name="cat" value="<?= $moveMovieData['catIndex'] ?>" />
            <input type="hidden" name="movie" value="<?= $moveMovieData['movieIndex'] ?>" />
            <div class="input-group">
                <label class="label" for="targetCat">Categoría destino:</label>
                <select name="targetCat" id="targetCat">
                    <?php foreach($jsonArray as $idx => $c): ?>
                        <?php if ($idx !== $moveMovieData['catIndex']): ?>
                            <option value="<?= $idx ?>"><?= htmlspecialchars($c['name']) ?></option>
                        <?php endif; ?>
                    <?php endforeach; ?>
                </select>
            </div>
            <button type="submit">Mover</button>
        </form>
    </div>
    <?php endif; ?>

    <!-- Bloque: Cambiar nombre de categoría -->
    <?php if ($editCategoryNameData): ?>
    <div class="block">
        <h2>Cambiar nombre de categoría: <?= htmlspecialchars($editCategoryNameData['name']) ?></h2>
        <form method="post" action="">
            <input type="hidden" name="action" value="updateCategoryName" />
            <input type="hidden" name="catIndex" value="<?= $editCategoryNameData['catIndex'] ?>" />
            <div class="input-group">
                <label class="label" for="newName">Nuevo nombre:</label>
                <input type="text" name="newName" id="newName" required value="<?= htmlspecialchars($editCategoryNameData['name']) ?>" />
            </div>
            <button type="submit">Guardar</button>
        </form>
    </div>
    <?php endif; ?>

    <!-- Bloque: Editar posición de categoría -->
    <?php if ($editCategoryPositionData): ?>
    <div class="block">
        <h2>Editar posición de categoría: <?= htmlspecialchars($editCategoryPositionData['name']) ?></h2>
        <form method="post" action="">
            <input type="hidden" name="action" value="updateCategory" />
            <input type="hidden" name="catIndex" value="<?= $editCategoryPositionData['catIndex'] ?>" />
            <div class="input-group">
                <label class="label" for="position">Nueva posición:</label>
                <?php $totalCats = count($jsonArray); ?>
                <select name="position" id="position" class="pos-field">
                    <?php for($i=1;$i<=$totalCats;$i++): ?>
                        <option value="<?= $i ?>" <?= $i==($editCategoryPositionData['catIndex']+1)?'selected':'' ?>><?= $i ?></option>
                    <?php endfor; ?>
                </select>
            </div>
            <button type="submit">Mover</button>
        </form>
    </div>
    <?php endif; ?>

    <!-- Bloque 3: Listado Actual de Categorías / Películas -->
    <div class="block category-list">
        <h2>3) Listado Actual de Categorías / Películas</h2>
        <?php if (empty($jsonArray)): ?>
            <p>No hay categorías cargadas.</p>
        <?php else: ?>
            <?php foreach ($jsonArray as $cIndex => $cat): ?>
                <h3>
                    <?= htmlspecialchars($cat['name']) ?>
                    <span class="move-links">
                        <a href="?action=moveCategory&index=<?= $cIndex ?>&dir=up">[↑]</a>
                        <a href="?action=moveCategory&index=<?= $cIndex ?>&dir=down">[↓]</a>
                    </span>
                    <a class="edit-link" href="?action=editCategoryName&cat=<?= $cIndex ?>">[Cambiar nombre]</a>
                    <a class="edit-link" href="?action=editCategoryPosition&cat=<?= $cIndex ?>">[Editar posición]</a>
                </h3>
                <?php if (empty($cat['samples'])): ?>
                    <p style="margin-left:20px;"><em>No hay películas en esta categoría.</em></p>
                <?php else: ?>
                    <ul>
                        <?php foreach ($cat['samples'] as $mIndex => $pel): ?>
                            <li>
                                <?= htmlspecialchars($pel['name']) ?>
                                <!-- Botón para editar -->
                                <a class="edit-link" href="?action=editMovie&cat=<?= $cIndex ?>&movie=<?= $mIndex ?>">Editar</a>
                                <!-- Botón para eliminar -->
                                <form method="get" style="display:inline" onsubmit="return confirm('¿Estás seguro de eliminar esta película?');">
                                    <input type="hidden" name="action" value="deleteMovie" />
                                    <input type="hidden" name="cat" value="<?= $cIndex ?>" />
                                    <input type="hidden" name="movie" value="<?= $mIndex ?>" />
                                    <button type="submit" class="delete-link">Eliminar</button>
                                </form>
                                <!-- Botón para mover a otra categoría -->
                                <form method="get" style="display:inline">
                                    <input type="hidden" name="action" value="showMoveMovieForm" />
                                    <input type="hidden" name="cat" value="<?= $cIndex ?>" />
                                    <input type="hidden" name="movie" value="<?= $mIndex ?>" />
                                    <button type="submit" class="edit-link">Agregar a otra categoría</button>
                                </form>
                                <!-- Controles de mover dentro de la misma -->
                                <span class="move-links">
                                    <a href="?action=moveMovie&cat=<?= $cIndex ?>&movie=<?= $mIndex ?>&dir=up">[↑]</a>
                                    <a href="?action=moveMovie&cat=<?= $cIndex ?>&movie=<?= $mIndex ?>&dir=down">[↓]</a>
                                </span>
                            </li>
                        <?php endforeach; ?>
                    </ul>
                <?php endif; ?>
            <?php endforeach; ?>
        <?php endif; ?>
    </div>
</div>
</body>
</html>
