<?php
// Autocarga de Composer (requiere vlucas/phpdotenv)
require __DIR__ . '/vendor/autoload.php';

use Dotenv\Dotenv;

// Carga variables de entorno
$dotenv = Dotenv::createImmutable(__DIR__);
$dotenv->load();

// Definir credenciales
define('PANEL_USER', $_ENV['PANEL_USER']);
define('PANEL_PASS', $_ENV['PANEL_PASS']);

// Iniciar sesión
session_start();

// NUEVO: Variables para almacenar mensajes sobre los banners
$banner_success_message = '';
if (isset($_SESSION['banner_success'])) {
    $banner_success_message = $_SESSION['banner_success'];
    unset($_SESSION['banner_success']);
}
$banner_error_message = '';
if (isset($_SESSION['banner_error'])) {
    $banner_error_message = $_SESSION['banner_error'];
    unset($_SESSION['banner_error']);
}

// Variables para almacenar mensajes sobre la actualización del Bearer
$bearer_success_message = '';
if (isset($_SESSION['bearer_success'])) {
    $bearer_success_message = $_SESSION['bearer_success'];
    unset($_SESSION['bearer_success']);
}
$bearer_error_message = '';
if (isset($_SESSION['bearer_error'])) {
    $bearer_error_message = $_SESSION['bearer_error'];
    unset($_SESSION['bearer_error']);
}

// Variable para almacenar errores del editor JSON
$json_editor_error = '';
if (isset($_SESSION['json_editor_error'])) {
    $json_editor_error = $_SESSION['json_editor_error'];
    unset($_SESSION['json_editor_error']);
}

// Procesar formulario de login
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['login'])) {
    $u = $_POST['user'] ?? '';
    $p = $_POST['pass'] ?? '';
    if ($u === PANEL_USER && $p === PANEL_PASS) {
        $_SESSION['logged_in'] = true;
        header('Location: ' . $_SERVER['PHP_SELF']);
        exit;
    } else {
        $error = 'Usuario o contraseña incorrectos.';
    }
}

// Si no está autenticado, mostramos login
if (empty($_SESSION['logged_in'])):
?>
<!DOCTYPE html>
<html lang="es">
<head>
  <meta charset="UTF-8">
  <title>Login - Panel Tv</title>
  <style>
    body{display:flex;align-items:center;justify-content:center;height:100vh;margin:0;background:#f2f2f2;}
    .box{background:#fff;padding:20px;border-radius:6px;box-shadow:0 0 10px rgba(0,0,0,0.1);}
    input{width:100%;padding:10px;margin:8px 0;border:1px solid #ccc;border-radius:4px;}
    button{width:100%;padding:10px;background:#007bff;color:#fff;border:none;border-radius:4px;cursor:pointer;}
    .error{color:#d9534f;margin-bottom:10px;}
  </style>
</head>
<body>
  <div class="box">
    <h2>Acceso Panel Tv</h2>
    <?php if(!empty($error)): ?><div class="error"><?= htmlspecialchars($error) ?></div><?php endif; ?>
    <form method="post">
      <input type="text" name="user" placeholder="Usuario" required autofocus>
      <input type="password" name="pass" placeholder="Contraseña" required>
      <button type="submit" name="login">Ingresar</button>
    </form>
  </div>
</body>
</html>
<?php
  exit;
endif;

/***************************************
 * Configuraciones iniciales y Encriptación
 **************************************/
// Ruta a los archivos JSON
define('JSON_FILE_PATH', __DIR__ . '/tv.json');
define('JSON_UNENCRYPTED_FILE_PATH', __DIR__ . '/tv2.json');
// NUEVO: Ruta al JSON y directorio de banners
define('BANNERS_JSON_PATH', __DIR__ . '/banners.json');
define('BANNERS_UPLOAD_DIR', __DIR__ . '/banners');
define('BANNERS_URL_PATH', 'banners'); // Nombre de la carpeta como se verá en la URL

// NUEVO: Ruta al archivo local del Bearer
define('BEARER_JSON_PATH', __DIR__ . '/bearer.json');

// Clave y método de encriptación AES
define('ENCRYPTION_KEY', 'e72of82ke0gu2o2k');
define('ENCRYPTION_METHOD', 'aes-128-ecb');

/****************************************************
 * FUNCIONES DE ENCRIPTACIÓN / DESENCRIPTACIÓN
 ****************************************************/
function encrypt_data($data) {
    if (empty($data)) return '';
    $encrypted = openssl_encrypt($data, ENCRYPTION_METHOD, ENCRYPTION_KEY, 0, '');
    return $encrypted;
}

function decrypt_data($data) {
    if (empty($data)) return '';
    $current_data = $data;
    for ($i = 0; $i < 2; $i++) {
        $decrypted_once = openssl_decrypt($current_data, ENCRYPTION_METHOD, ENCRYPTION_KEY, 0, '');
        if ($decrypted_once === false) return $current_data;
        $current_data = $decrypted_once;
    }
    return $current_data;
}

function decrypt_json_data_recursively(&$data) {
    if (!is_array($data)) return;
    foreach ($data as &$section) {
        foreach (['samples', 'hidden_samples'] as $collectionKey) {
            if (isset($section[$collectionKey]) && is_array($section[$collectionKey])) {
                foreach ($section[$collectionKey] as &$sample) {
                    if (isset($sample['url'])) $sample['url'] = decrypt_data($sample['url']);
                    if (isset($sample['drm_license_uri'])) $sample['drm_license_uri'] = decrypt_data($sample['drm_license_uri']);
                    if (isset($sample['headers']) && is_array($sample['headers'])) {
                        foreach ($sample['headers'] as &$value) $value = decrypt_data($value);
                    }
                }
            }
        }
    }
}

function encrypt_json_data_recursively(&$data) {
    if (!is_array($data)) return;
    foreach ($data as &$section) {
        foreach (['samples', 'hidden_samples'] as $collectionKey) {
            if (isset($section[$collectionKey]) && is_array($section[$collectionKey])) {
                foreach ($section[$collectionKey] as &$sample) {
                    if (isset($sample['url'])) $sample['url'] = encrypt_data($sample['url']);
                    if (isset($sample['drm_license_uri'])) $sample['drm_license_uri'] = encrypt_data($sample['drm_license_uri']);
                    if (isset($sample['headers']) && is_array($sample['headers'])) {
                        foreach ($sample['headers'] as &$value) $value = encrypt_data($value);
                    }
                }
            }
        }
    }
}

/****************************************************
 * FUNCIONES DE MANEJO DE ARCHIVO JSON LOCAL
 ****************************************************/
function getJsonFromFile($filePath) {
    if (!file_exists($filePath)) {
        if ($filePath === JSON_FILE_PATH || $filePath === BANNERS_JSON_PATH || $filePath === BEARER_JSON_PATH) {
            // NUEVO: Crear archivos con estructura inicial si no existen
            if ($filePath === BANNERS_JSON_PATH) {
                $defaultContent = '{"events": []}';
            } elseif ($filePath === BEARER_JSON_PATH) {
                $defaultContent = '{"bearerToken": ""}';
            } else {
                $defaultContent = '[]';
            }
            file_put_contents($filePath, $defaultContent);
        }
        return json_decode($defaultContent ?? '[]', true);
    }
    $jsonContent = file_get_contents($filePath);
    $data = json_decode($jsonContent, true);
    return is_array($data) ? $data : [];
}

function updateJsonInFile($filePath, $dataArray) {
    $updatedJson = json_encode($dataArray, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    if (file_put_contents($filePath, $updatedJson) === false) {
        die("Error: No se pudo escribir en el archivo '{$filePath}'. Verifica los permisos.");
    }
}

/****************************************************
 * NUEVAS FUNCIONES PARA DETECTAR JSON SIN ENCRIPTAR
 ****************************************************/
function is_base64_encoded($data) {
    if (!is_string($data) || empty(trim($data))) return false;
    return (base64_decode($data, true) !== false);
}

function checkForUnencryptedData($filePath) {
    $data = getJsonFromFile($filePath);
    if (empty($data)) return false;
    foreach ($data as $section) {
        foreach (['samples', 'hidden_samples'] as $collectionKey) {
            if (empty($section[$collectionKey])) continue;
            foreach ($section[$collectionKey] as $sample) {
                if (isset($sample['url']) && !is_base64_encoded($sample['url'])) return true;
                if (isset($sample['drm_license_uri']) && !is_base64_encoded($sample['drm_license_uri'])) return true;
                if (isset($sample['headers']) && is_array($sample['headers'])) {
                    foreach ($sample['headers'] as $value) if (!is_base64_encoded($value)) return true;
                }
            }
        }
    }
    return false;
}

/****************************************************
 * LÓGICA DE REORDENAMIENTO (CATEGORÍAS Y CANALES)
 ****************************************************/
function moveCategoryPosition(&$data, $indexFrom, $indexTo) {
    $indexFrom = intval($indexFrom);
    $indexTo   = intval($indexTo);
    if ($indexFrom < 0 || $indexFrom >= count($data) || $indexTo < 0 || $indexTo >= count($data)) return;
    $item = array_splice($data, $indexFrom, 1);
    array_splice($data, $indexTo, 0, $item);
}

function moveVisibleSection(&$data, $sectionIndex, $targetVisiblePos) {
    $sectionIndex = intval($sectionIndex);
    $targetVisiblePos = intval($targetVisiblePos);

    if ($sectionIndex < 0 || $sectionIndex >= count($data)) {
        return;
    }

    $visibleIndices = [];
    foreach ($data as $idx => $section) {
        if (empty($section['hidden'])) {
            $visibleIndices[] = $idx;
        }
    }

    $currentVisiblePos = array_search($sectionIndex, $visibleIndices, true);
    if ($currentVisiblePos === false) {
        return; // La sección ya está oculta
    }

    if ($targetVisiblePos < 0 || $targetVisiblePos >= count($visibleIndices)) {
        return;
    }

    if ($currentVisiblePos === $targetVisiblePos) {
        return;
    }

    $targetSectionIndex = $visibleIndices[$targetVisiblePos];

    $section = $data[$sectionIndex];
    array_splice($data, $sectionIndex, 1);

    if ($sectionIndex < $targetSectionIndex) {
        $targetSectionIndex--;
    }

    array_splice($data, $targetSectionIndex, 0, [$section]);
}

function moveChannelPosition(&$samples, $oldIndex, $newIndex) {
    if ($oldIndex === $newIndex) return;
    if ($oldIndex < 0 || $oldIndex >= count($samples) || $newIndex < 0 || $newIndex >= count($samples)) return;
    $item = array_splice($samples, $oldIndex, 1);
    array_splice($samples, $newIndex, 0, $item);
}

/****************************************************
 * NUEVA FUNCIÓN: Subida de logos/íconos de canal a /banners
 ****************************************************/
function upload_image_to_banners($fileFieldName) {
    // Usa la misma carpeta /banners de los banners superiores
    if (!isset($_FILES[$fileFieldName]) || !is_array($_FILES[$fileFieldName])) return [false, ''];
    if ($_FILES[$fileFieldName]['error'] !== 0) return [false, ''];

    // Validar directorio destino
    if (!is_dir(BANNERS_UPLOAD_DIR) || !is_writable(BANNERS_UPLOAD_DIR)) {
        return [false, ''];
    }

    // Validar tipo de archivo
    $allowed = ['jpg', 'jpeg', 'png', 'gif', 'webp'];
    $filename = $_FILES[$fileFieldName]['name'];
    $ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
    if (!in_array($ext, $allowed)) return [false, ''];

    // Nombre único
    $newFileName = uniqid('', true) . '.' . $ext;
    $destination = BANNERS_UPLOAD_DIR . '/' . $newFileName;

    // Mover archivo
    if (!move_uploaded_file($_FILES[$fileFieldName]['tmp_name'], $destination)) {
        return [false, ''];
    }

    // Armar URL pública
    $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? "https" : "http";
    $host = $_SERVER['HTTP_HOST'];
    $uri = rtrim(dirname($_SERVER['REQUEST_URI']), '/\\');
    $publicUrl = "$protocol://$host$uri/" . BANNERS_URL_PATH . "/$newFileName";

    return [true, $publicUrl];
}

/****************************************************
 * OBTENER DATOS INICIALES
 ****************************************************/
$data = getJsonFromFile(JSON_FILE_PATH);
// NUEVO: Obtener datos de banners y verificar directorio
$bannersData = getJsonFromFile(BANNERS_JSON_PATH);
$bannersDirError = '';
if (!is_dir(BANNERS_UPLOAD_DIR) || !is_writable(BANNERS_UPLOAD_DIR)) {
    $bannersDirError = "¡Advertencia! El directorio de banners no existe o no se puede escribir en él. Por favor, crea una carpeta llamada '<strong>" . BANNERS_URL_PATH . "</strong>' en el mismo directorio que este panel y asegúrate de que tenga permisos de escritura.";
}

$unencryptedDataDetected = checkForUnencryptedData(JSON_UNENCRYPTED_FILE_PATH);


/****************************************************
 * PROCESAR ACCIONES DEL USUARIO (POST)
 ****************************************************/
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';

    // NUEVO: Bloque de switch para acciones de banners
    switch ($action) {
        case 'add_banner':
            if (isset($_FILES['banner_file']) && $_FILES['banner_file']['error'] == 0) {
                if ($bannersDirError) { // Re-verificar antes de subir
                    $_SESSION['banner_error'] = $bannersDirError;
                } else {
                    $allowed = ['jpg', 'jpeg', 'png', 'gif', 'webp'];
                    $filename = $_FILES['banner_file']['name'];
                    $filetype = pathinfo($filename, PATHINFO_EXTENSION);
                    if (!in_array(strtolower($filetype), $allowed)) {
                        $_SESSION['banner_error'] = 'Error: Tipo de archivo no permitido. Solo se aceptan: ' . implode(', ', $allowed);
                    } else {
                        $newFileName = uniqid('', true) . '.' . $filetype;
                        $destination = BANNERS_UPLOAD_DIR . '/' . $newFileName;
                        
                        if (move_uploaded_file($_FILES['banner_file']['tmp_name'], $destination)) {
                            // Construir URL pública completa
                            $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? "https" : "http";
                            $host = $_SERVER['HTTP_HOST'];
                            $uri = rtrim(dirname($_SERVER['REQUEST_URI']), '/\\');
                            $newBannerUrl = "$protocol://$host$uri/" . BANNERS_URL_PATH . "/$newFileName";
                            
                            $bannersData['events'][] = $newBannerUrl;
                            updateJsonInFile(BANNERS_JSON_PATH, $bannersData);
                            $_SESSION['banner_success'] = '¡Banner subido y agregado correctamente!';
                        } else {
                            $_SESSION['banner_error'] = 'Error: No se pudo mover el archivo subido.';
                        }
                    }
                }
            } else {
                $_SESSION['banner_error'] = 'Error al subir el archivo. Código de error: ' . ($_FILES['banner_file']['error'] ?? 'N/A');
            }
            break;

        case 'delete_banner':
            $bannerIndex = intval($_POST['banner_index']);
            if (isset($bannersData['events'][$bannerIndex])) {
                $urlToDelete = $bannersData['events'][$bannerIndex];
                $filenameToDelete = basename($urlToDelete);
                $filePathToDelete = BANNERS_UPLOAD_DIR . '/' . $filenameToDelete;

                if (file_exists($filePathToDelete)) {
                    unlink($filePathToDelete);
                }

                array_splice($bannersData['events'], $bannerIndex, 1);
                updateJsonInFile(BANNERS_JSON_PATH, $bannersData);
                $_SESSION['banner_success'] = 'Banner eliminado correctamente.';
            } else {
                $_SESSION['banner_error'] = 'Error: No se pudo encontrar el banner a eliminar.';
            }
            break;

        case 'reorder_banners':
            $newOrder = $_POST['banner_order'] ?? [];
            if (!empty($newOrder)) {
                $bannersData['events'] = $newOrder;
                updateJsonInFile(BANNERS_JSON_PATH, $bannersData);
                $_SESSION['banner_success'] = '¡Orden de los banners actualizado!';
            }
            break;
    }
    
    // Switch para acciones del panel de TV
    switch ($action) {
        case 'update_bearer':
            $newBearer = $_POST['new_bearer_token'] ?? '';
            $trimmedBearer = trim($newBearer);

            if (empty($trimmedBearer)) {
                 $_SESSION['bearer_error'] = 'Error: El campo del token no puede estar vacío.';
            } elseif (strpos($trimmedBearer, 'Bearer ') !== 0) {
                 $_SESSION['bearer_error'] = 'Error: El token debe comenzar con "Bearer " (con un espacio al final).';
            } elseif (count(explode(' ', $trimmedBearer)) > 2) {
                 $_SESSION['bearer_error'] = 'Error: Se detectaron espacios extra. El formato debe ser "Bearer [token_sin_espacios]".';
            } else {
                // NUEVO: Guardar localmente en bearer.json (mismo directorio del panel)
                $payload = json_encode(['bearerToken' => $trimmedBearer], JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
                $result = @file_put_contents(BEARER_JSON_PATH, $payload);
                if ($result !== false) {
                    $_SESSION['bearer_success'] = '¡Bearer token actualizado en bearer.json correctamente!';
                } else {
                    $_SESSION['bearer_error'] = 'Error al escribir en bearer.json. Verifica permisos de escritura en el directorio.';
                }
            }
            break;

        case 'save_raw_json':
            $rawJsonContent = $_POST['json_content'] ?? '[]';
            $decodedData = json_decode($rawJsonContent, true);
            if (json_last_error() !== JSON_ERROR_NONE) {
                $_SESSION['json_editor_error'] = "Error de sintaxis en el JSON: " . json_last_error_msg() . ". No se pudo guardar.";
                $_SESSION['json_editor_content_buffer'] = $rawJsonContent; 
                header("Location: " . $_SERVER['PHP_SELF'] . "?view=editor"); exit;
            }
            encrypt_json_data_recursively($decodedData);
            updateJsonInFile(JSON_FILE_PATH, $decodedData);
            header("Location: " . $_SERVER['PHP_SELF']); exit;
            break;

        case 'encrypt_and_merge':
            $dataToProcess = getJsonFromFile(JSON_UNENCRYPTED_FILE_PATH);
            if (!empty($dataToProcess)) {
                foreach ($dataToProcess as &$section) {
                    foreach (['samples', 'hidden_samples'] as $collectionKey) {
                        if (empty($section[$collectionKey])) continue;
                        foreach ($section[$collectionKey] as &$sample) {
                            if (isset($sample['url'])) $sample['url'] = encrypt_data(decrypt_data($sample['url']));
                            if (isset($sample['drm_license_uri'])) $sample['drm_license_uri'] = encrypt_data(decrypt_data($sample['drm_license_uri']));
                            if (isset($sample['headers']) && is_array($sample['headers'])) {
                                foreach ($sample['headers'] as &$value) $value = encrypt_data(decrypt_data($value));
                            }
                        }
                    }
                }
                updateJsonInFile(JSON_FILE_PATH, $dataToProcess);
            }
            break;

        case 'add_section':
            $newSectionName = trim($_POST['section_name']);
            if ($newSectionName !== '') $data[] = ["name" => $newSectionName, "samples" => []];
            break;

        case 'rename_section':
            $sectionIndex = intval($_POST['section_index']);
            $newSectionName = trim($_POST['new_section_name']);
            if (isset($data[$sectionIndex]) && $newSectionName !== '') $data[$sectionIndex]['name'] = $newSectionName;
            break;

        case 'hide_section':
            $sectionIndex = intval($_POST['section_index']);
            if (isset($data[$sectionIndex])) {
                $data[$sectionIndex]['hidden'] = true;
            }
            break;

        case 'show_section':
            $sectionIndex = intval($_POST['section_index']);
            if (isset($data[$sectionIndex])) {
                unset($data[$sectionIndex]['hidden']);
            }
            break;

        case 'delete_section':
            $sectionIndex = intval($_POST['section_index']);
            if (isset($data[$sectionIndex])) array_splice($data, $sectionIndex, 1);
            break;

        case 'move_section':
            $sectionIndex = intval($_POST['section_index']);
            $newPosition  = intval($_POST['new_position']) - 1;
            moveVisibleSection($data, $sectionIndex, $newPosition);
            break;

        case 'add_channel':
            $sectionIndex = intval($_POST['section_index']);
            if (isset($data[$sectionIndex])) {
                // Determinar URL del icono: prioriza archivo subido
                $iconFromText = trim($_POST['channel_icon'] ?? '');
                $iconFinal = $iconFromText;
                if (isset($_FILES['channel_icon_file']) && $_FILES['channel_icon_file']['error'] === 0) {
                    list($okUpload, $uploadedUrl) = upload_image_to_banners('channel_icon_file');
                    if ($okUpload) $iconFinal = $uploadedUrl;
                }

                $newChannel = [
                    "name"  => trim($_POST['channel_name']),
                    "url"   => encrypt_data(trim($_POST['channel_url'])),
                    "type"  => $_POST['channel_type'],
                    "icono" => $iconFinal
                ];
                if ($newChannel['type'] === "CLEARKEY") $newChannel["drm_license_uri"] = encrypt_data(trim($_POST['channel_drm']));
                if (($_POST['has_headers'] ?? 'no') === 'yes') {
                    $headersObj = [];
                    foreach ($_POST['channel_header_key'] ?? [] as $i => $k) {
                        $v = $_POST['channel_header_val'][$i] ?? '';
                        if (trim($k) !== '') $headersObj[trim($k)] = encrypt_data(trim($v));
                    }
                    if (!empty($headersObj)) $newChannel["headers"] = $headersObj;
                }
                $data[$sectionIndex]['samples'][] = $newChannel;
            }
            break;

        case 'delete_channel':
            $sectionIndex = intval($_POST['section_index']);
            $channelIndex = intval($_POST['channel_index']);
            $channelSource = $_POST['channel_source'] ?? 'visible';
            $collectionKey = ($channelSource === 'hidden') ? 'hidden_samples' : 'samples';
            if (isset($data[$sectionIndex][$collectionKey][$channelIndex])) {
                array_splice($data[$sectionIndex][$collectionKey], $channelIndex, 1);
                if ($collectionKey === 'hidden_samples' && empty($data[$sectionIndex][$collectionKey])) {
                    unset($data[$sectionIndex][$collectionKey]);
                }
            }
            break;

        case 'hide_channel':
            $sectionIndex = intval($_POST['section_index']);
            $channelIndex = intval($_POST['channel_index']);
            if (isset($data[$sectionIndex]['samples'][$channelIndex])) {
                $channel = $data[$sectionIndex]['samples'][$channelIndex];
                array_splice($data[$sectionIndex]['samples'], $channelIndex, 1);
                if (!isset($data[$sectionIndex]['hidden_samples']) || !is_array($data[$sectionIndex]['hidden_samples'])) {
                    $data[$sectionIndex]['hidden_samples'] = [];
                }
                $data[$sectionIndex]['hidden_samples'][] = $channel;
            }
            break;

        case 'unhide_channel':
            $sectionIndex = intval($_POST['section_index']);
            $channelIndex = intval($_POST['channel_index']);
            if (isset($data[$sectionIndex]['hidden_samples'][$channelIndex])) {
                $channel = $data[$sectionIndex]['hidden_samples'][$channelIndex];
                array_splice($data[$sectionIndex]['hidden_samples'], $channelIndex, 1);
                if (empty($data[$sectionIndex]['hidden_samples'])) unset($data[$sectionIndex]['hidden_samples']);
                if (!isset($data[$sectionIndex]['samples']) || !is_array($data[$sectionIndex]['samples'])) {
                    $data[$sectionIndex]['samples'] = [];
                }
                $data[$sectionIndex]['samples'][] = $channel;
            }
            break;

        case 'edit_channel':
            $sectionIndex = intval($_POST['section_index']);
            $channelIndex = intval($_POST['channel_index']);
            $channelSource = $_POST['channel_source'] ?? 'visible';
            $collectionKey = ($channelSource === 'hidden') ? 'hidden_samples' : 'samples';
            if (isset($data[$sectionIndex][$collectionKey][$channelIndex])) {
                $channel =& $data[$sectionIndex][$collectionKey][$channelIndex];

                // Determinar URL del icono: prioriza archivo subido si llega
                $iconFromText = trim($_POST['channel_icon'] ?? '');
                $iconFinal = $iconFromText;
                if (isset($_FILES['channel_icon_file']) && $_FILES['channel_icon_file']['error'] === 0) {
                    list($okUpload, $uploadedUrl) = upload_image_to_banners('channel_icon_file');
                    if ($okUpload) $iconFinal = $uploadedUrl;
                }

                $channel['name']  = trim($_POST['channel_name']);
                $channel['url']   = encrypt_data(trim($_POST['channel_url']));
                $channel['icono'] = $iconFinal;
                $channel['type']  = $_POST['channel_type'];

                if ($channel['type'] === "CLEARKEY") $channel['drm_license_uri'] = encrypt_data(trim($_POST['channel_drm']));
                else unset($channel['drm_license_uri']);
                if (($_POST['has_headers'] ?? 'no') === 'yes') {
                    $headersObj = [];
                    foreach ($_POST['channel_header_key'] ?? [] as $i => $k) {
                        $v = $_POST['channel_header_val'][$i] ?? '';
                        if (trim($k) !== '') $headersObj[trim($k)] = encrypt_data(trim($v));
                    }
                    $channel['headers'] = $headersObj;
                } else unset($channel['headers']);
                if ($channelSource === 'visible' && isset($_POST['channel_new_position'])) {
                    $channelNewPosition = intval($_POST['channel_new_position']) - 1;
                    moveChannelPosition($data[$sectionIndex]['samples'], $channelIndex, $channelNewPosition);
                }
            }
            break;
    }

    // Para todas las acciones del panel de TV, actualizamos tv.json
    $tv_actions = ['add_section', 'rename_section', 'delete_section', 'move_section', 'add_channel', 'delete_channel', 'edit_channel', 'encrypt_and_merge', 'hide_channel', 'unhide_channel', 'hide_section', 'show_section'];
    if (in_array($action, $tv_actions)) {
        updateJsonInFile(JSON_FILE_PATH, $data);
    }
    
    // Redirigir para evitar reenvío de formulario
    header("Location: " . $_SERVER['PHP_SELF']);
    exit;
}

$visibleSections = [];
$hiddenSections = [];
foreach ($data as $idx => $section) {
    if (!empty($section['hidden'])) {
        $hiddenSections[$idx] = $section;
    } else {
        $visibleSections[$idx] = $section;
    }
}
$visibleSectionKeys = array_keys($visibleSections);

// Preparar contenido para el editor JSON si está activo
$json_for_editor = '';
if (isset($_SESSION['json_editor_content_buffer'])) {
    $json_for_editor = $_SESSION['json_editor_content_buffer'];
    unset($_SESSION['json_editor_content_buffer']);
} else {
    $data_for_editor = getJsonFromFile(JSON_FILE_PATH);
    decrypt_json_data_recursively($data_for_editor);
    $json_for_editor = json_encode($data_for_editor, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <title>Panel de Administración</title>
    <style>
        body { background: #f0f2f5; margin: 0; padding: 0; font-family: sans-serif; }
        .container { width: 90%; max-width: 1100px; margin: 20px auto; }
        .panel-section { background: #fff; padding: 20px; border-radius: 8px; margin-bottom: 25px; box-shadow: 0 2px 4px rgba(0,0,0,0.1); }
        h1, h2, h3 { text-align: center; margin: 10px 0 20px 0; }
        .section-box { border: 1px solid #ccc; padding: 10px; margin-bottom: 20px; border-radius: 5px; }
        .section-header { display: flex; justify-content: space-between; align-items: flex-start; }
        .section-title { display: inline-block; margin: 0; }
        .channels-list { margin-left: 20px; margin-top: 10px; list-style: none; padding-left: 0; }
        .channels-list li { margin-bottom: 8px; padding: 5px; border-left: 3px solid #eee; }
        .channels-list li.hidden-divider { border: none; margin-top: 12px; font-style: italic; color: #777; padding-left: 0; }
        .channels-list li.channel-hidden { background: #fff7e6; border-left-color: #ff9800; opacity: 0.8; }
        .badge-oculto { background: #ff9800; color: #fff; padding: 2px 8px; border-radius: 999px; font-size: 0.75em; margin-left: 6px; }
        .note-hidden-edit { font-size: 0.85em; color: #555; margin-top: 8px; }
        .section-hidden-box { border: 1px dashed #ff9800; background: #fff7e6; padding: 15px; border-radius: 5px; margin-top: 20px; }
        .section-hidden-box ul { margin-top: 10px; }
        button { cursor: pointer; margin: 2px; }
        form { margin: 0; padding: 0; }
        .inline-form { display: inline-block; }
        .label-input { margin: 6px 0; }
        .input-text { width: 95%; padding: 4px; margin: 4px 0; }
        .header-pair { display: flex; gap: 5px; margin-bottom: 5px; }
        .header-pair input { width: 45%; }
        .add-header-btn { margin: 8px 0; }
        .small { font-size: 0.9em; color: #666; }
        .btn-primary { background: #28a745; color: #fff; border: none; padding: 5px 10px; border-radius: 3px; }
        .btn-danger { background: #dc3545; color: #fff; border: none; padding: 5px 10px; border-radius: 3px; }
        .btn-secondary { background: #007bff; color: #fff; border: none; padding: 5px 10px; border-radius: 3px; }
        .btn-warning { background: #ffc107; color: #333; border: none; padding: 5px 10px; border-radius: 3px; }
        .add-section-form, .add-channel-form, .edit-channel-form { background: #f7f7f7; margin-top: 15px; padding: 10px; border-radius: 5px; border: 1px solid #e0e0e0; }
        .alert-unencrypted, .alert-success, .alert-danger, .alert-warning { padding: 15px; margin-bottom: 20px; border-radius: 5px; text-align: center; }
        .alert-unencrypted { background-color: #fff3cd; border: 1px solid #ffeeba; color: #856404; }
        .alert-warning { background-color: #fff3cd; border: 1px solid #ffeeba; color: #856404; }
        .alert-success { background-color: #d4edda; border: 1px solid #c3e6cb; color: #155724; }
        .alert-danger { background-color: #f8d7da; border: 1px solid #f5c6cb; color: #721c24; }
        .top-controls { display: flex; justify-content: space-between; align-items: center; margin-bottom: 15px; border-bottom: 1px solid #ddd; padding-bottom: 10px; }
        .top-controls div:first-child { flex-grow: 1; text-align: center; }
        .top-controls div:last-child { display: flex; gap: 10px; }
        #jsonEditorContainer { display: none; }
        .json-editor-warning { background-color: #f8d7da; border: 1px solid #f5c6cb; color: #721c24; padding: 15px; margin-bottom: 15px; border-radius: 5px; text-align: center; font-weight: bold; }
        #json_content_editor { width: 100%; height: 60vh; font-family: 'Courier New', Courier, monospace; font-size: 14px; border: 1px solid #ccc; border-radius: 4px; padding: 10px; box-sizing: border-box; }
        #findReplaceBar { display: none; background: #f0f0f0; padding: 8px; margin-bottom: 10px; border-radius: 4px; }
        #findReplaceBar input[type="text"] { margin-right: 5px; }
        #findReplaceBar button { margin: 0 2px; }
        #syntaxError { color: #d9534f; margin-bottom: 10px; font-weight: bold; }
        .modal-overlay { display: none; position: fixed; z-index: 1000; left: 0; top: 0; width: 100%; height: 100%; overflow: auto; background-color: rgba(0,0,0,0.4); }
        .modal-content { background-color: #fefefe; margin: 15% auto; padding: 20px; border: 1px solid #888; width: 80%; max-width: 500px; border-radius: 8px; box-shadow: 0 4px 8px 0 rgba(0,0,0,0.2); }
        .modal-content textarea { width: 100%; height: 100px; padding: 10px; box-sizing: border-box; margin-bottom: 15px; border-radius: 4px; border: 1px solid #ccc; }
        .modal-buttons { text-align: right; }

        /* NUEVO: Estilos para la sección de Banners */
        .upload-banner-form { background: #f7f7f7; padding: 15px; border: 1px solid #e0e0e0; border-radius: 5px; margin-bottom: 20px; }
        .banner-grid { display: grid; grid-template-columns: repeat(auto-fill, minmax(200px, 1fr)); gap: 15px; margin-top: 15px; }
        .banner-item { border: 1px solid #ddd; border-radius: 4px; padding: 10px; text-align: center; position: relative; cursor: move; background: #fff; }
        .banner-item.dragging { opacity: 0.6; }
        .banner-item img { max-width: 100%; height: 100px; object-fit: cover; border-radius: 4px; margin-bottom: 8px; }
        .banner-item .delete-form { position: absolute; top: 5px; right: 5px; }
        .banner-item .url-text { font-size: 0.7em; word-break: break-all; color: #666; }
        .reorder-form-submit { text-align: right; margin-top: 15px; }
    </style>
</head>
<body>
<div class="container">

    <!-- NUEVO: SECCIÓN DE GESTIÓN DE BANNERS -->
    <div class="panel-section">
        <h2>Gestión de Banners</h2>
        <p class="small" style="text-align:center;">Editando archivo: <strong>banners.json</strong> | Las imágenes se suben a la carpeta: <strong>/banners</strong></p>

        <?php if ($banner_success_message): ?><div class="alert-success"><?php echo htmlspecialchars($banner_success_message); ?></div><?php endif; ?>
        <?php if ($banner_error_message): ?><div class="alert-danger"><?php echo htmlspecialchars($banner_error_message); ?></div><?php endif; ?>
        <?php if ($bannersDirError): ?><div class="alert-danger"><?php echo $bannersDirError; ?></div><?php endif; ?>

        <div class="upload-banner-form">
            <h3>Subir nuevo Banner</h3>
            <form method="POST" enctype="multipart/form-data">
                <input type="hidden" name="action" value="add_banner">
                <input type="file" name="banner_file" required accept="image/png, image/jpeg, image/gif, image/webp">
                <button type="submit" class="btn-primary">Subir Banner</button>
            </form>
        </div>

        <h3>Banners Actuales (Arrastra para reordenar)</h3>
        <div id="banner-list" class="banner-grid">
                <?php if (!empty($bannersData['events'])): ?>
                    <?php foreach ($bannersData['events'] as $index => $bannerUrl): ?>
                        <div class="banner-item" draggable="true" data-url="<?php echo htmlspecialchars($bannerUrl); ?>">
                            <img src="<?php echo htmlspecialchars($bannerUrl); ?>" alt="Banner <?php echo $index + 1; ?>">
                            <p class="url-text"><?php echo htmlspecialchars(basename($bannerUrl)); ?></p>
                            <!-- Formulario para eliminar -->
                            <form method="POST" class="delete-form inline-form">
                                <input type="hidden" name="action" value="delete_banner">
                                <input type="hidden" name="banner_index" value="<?php echo $index; ?>">
                                <button type="submit" class="btn-danger" onclick="return confirm('¿Estás seguro de eliminar este banner? Se borrará el archivo permanentemente.')">X</button>
                            </form>
                        </div>
                    <?php endforeach; ?>
                <?php else: ?>
                    <p>No hay banners para mostrar. ¡Sube uno!</p>
                <?php endif; ?>
        </div>
        <form method="POST" id="reorder-banners-form">
            <input type="hidden" name="action" value="reorder_banners">
            <div id="banner-order-inputs"></div>
            <div class="reorder-form-submit">
                <button type="submit" class="btn-primary">Guardar Orden</button>
            </div>
        </form>
    </div>


    <div id="panelNormal" class="panel-section">
        <div class="top-controls">
            <div><h1>Panel de Administración - TV JSON</h1></div>
            <div>
                <button id="updateBearerBtn" class="btn-warning">Actualizar Bearer</button>
                <button id="showJsonEditorBtn" class="btn-secondary">Ver JSON Normal</button>
            </div>
        </div>
        <p class="small" style="text-align:center;">Editando archivo local: <strong>tv.json</strong> | Bearer local: <strong><?php echo htmlspecialchars(basename(BEARER_JSON_PATH)); ?></strong></p>
        
        <?php if ($bearer_success_message): ?><div class="alert-success"><?php echo htmlspecialchars($bearer_success_message); ?></div><?php endif; ?>
        <?php if ($bearer_error_message): ?><div class="alert-danger"><?php echo $bearer_error_message; ?></div><?php endif; ?>

        <?php if ($unencryptedDataDetected): ?>
        <div class="alert-unencrypted">
            <strong>¡Alerta!</strong> Se ha detectado el archivo <strong>tv2.json</strong> con contenido sin encriptar.
            <br>Presiona el botón para encriptar todo su contenido y cargarlo en el archivo principal (tv.json).
            <form method="POST" style="margin-top:10px;">
                <input type="hidden" name="action" value="encrypt_and_merge">
                <button type="submit" class="btn-warning">Encriptar y Cargar desde tv2.json</button>
            </form>
        </div>
        <?php endif; ?>

        <div class="add-section-form">
            <h3>Agregar nueva categoría</h3>
            <form method="POST">
                <input type="hidden" name="action" value="add_section">
                <div class="label-input">
                    <label>Nombre de la sección:</label><br>
                    <input type="text" name="section_name" class="input-text" required>
                </div>
                <button type="submit" class="btn-primary">Agregar sección</button>
            </form>
        </div>

        <?php foreach ($visibleSections as $sectionIndex => $section): ?>
            <div class="section-box">
                <div class="section-header">
                    <div>
                        <h2 class="section-title" style="display:inline;"><?php echo htmlspecialchars($section['name']); ?></h2>
                        <button type="button" class="btn-secondary" onclick="toggleRenameForm('<?php echo $sectionIndex; ?>')">Cambiar nombre</button>
                        <div id="renameForm_<?php echo $sectionIndex; ?>" style="display:none; margin-top:5px;">
                            <form method="POST" class="inline-form">
                                <input type="hidden" name="action" value="rename_section"><input type="hidden" name="section_index" value="<?php echo $sectionIndex; ?>">
                                <input type="text" name="new_section_name" value="<?php echo htmlspecialchars($section['name']); ?>" class="input-text" required>
                                <button type="submit" class="btn-primary">Actualizar</button>
                            </form>
                        </div>
                    </div>
                    <div>
                        <form method="POST" class="inline-form">
                            <input type="hidden" name="action" value="hide_section"><input type="hidden" name="section_index" value="<?php echo $sectionIndex; ?>">
                            <button type="submit" class="btn-secondary" onclick="return confirm('¿Ocultar esta sección completa?');">Ocultar</button>
                        </form>
                        <form method="POST" class="inline-form">
                            <input type="hidden" name="action" value="delete_section"><input type="hidden" name="section_index" value="<?php echo $sectionIndex; ?>">
                            <button type="submit" class="btn-danger" onclick="return confirm('¿Estás seguro de eliminar la sección completa?')">Eliminar</button>
                        </form>
                        <form method="POST" class="inline-form">
                            <input type="hidden" name="action" value="move_section"><input type="hidden" name="section_index" value="<?php echo $sectionIndex; ?>">
                            <?php $visibleOrder = array_search($sectionIndex, $visibleSectionKeys, true); $visibleOrder = ($visibleOrder === false) ? 0 : $visibleOrder; ?>
                            <label>Posición:</label> <input type="number" name="new_position" value="<?php echo ($visibleOrder + 1); ?>" min="1" max="<?php echo count($visibleSectionKeys); ?>" style="width:60px;">
                            <button type="submit" class="btn-warning">Mover</button>
                        </form>
                    </div>
                </div>

                <ul class="channels-list">
                    <?php 
                        $visibleChannels = $section['samples'] ?? [];
                        $hiddenChannels  = $section['hidden_samples'] ?? [];
                    ?>
                    <?php if (!empty($visibleChannels) || !empty($hiddenChannels)): ?>
                        <?php foreach ($visibleChannels as $channelIndex => $channel): ?>
                            <li>
                                <strong><?php echo htmlspecialchars($channel['name']); ?></strong>
                                <form method="POST" class="inline-form">
                                    <input type="hidden" name="action" value="delete_channel"><input type="hidden" name="section_index" value="<?php echo $sectionIndex; ?>"><input type="hidden" name="channel_index" value="<?php echo $channelIndex; ?>"><input type="hidden" name="channel_source" value="visible">
                                    <button type="submit" class="btn-danger" onclick="return confirm('¿Eliminar este canal?')">Eliminar</button>
                                </form>
                                <form method="POST" class="inline-form">
                                    <input type="hidden" name="action" value="hide_channel"><input type="hidden" name="section_index" value="<?php echo $sectionIndex; ?>"><input type="hidden" name="channel_index" value="<?php echo $channelIndex; ?>">
                                    <button type="submit" class="btn-secondary">Ocultar</button>
                                </form>
                                <button type="button" class="btn-secondary" onclick="toggleEditForm('<?php echo $sectionIndex; ?>','<?php echo $channelIndex; ?>','visible')">Editar</button>

                                <div id="editForm_visible_<?php echo $sectionIndex; ?>_<?php echo $channelIndex; ?>" class="edit-channel-form" style="display:none;">
                                    <h4>Editar canal</h4>
                                    <form method="POST" enctype="multipart/form-data">
                                        <input type="hidden" name="action" value="edit_channel"><input type="hidden" name="section_index" value="<?php echo $sectionIndex; ?>"><input type="hidden" name="channel_index" value="<?php echo $channelIndex; ?>"><input type="hidden" name="channel_source" value="visible">
                                        <div class="label-input"><label>Nombre:</label><br><input type="text" name="channel_name" class="input-text" value="<?php echo htmlspecialchars($channel['name']); ?>" required></div>
                                        <div class="label-input"><label>URL del canal:</label><br><input type="text" name="channel_url" class="input-text" value="<?php echo htmlspecialchars(decrypt_data($channel['url'])); ?>"></div>
                                        <div class="label-input">
                                            <label>URL del icono:</label><br>
                                            <input type="text" name="channel_icon" class="input-text" value="<?php echo htmlspecialchars($channel['icono'] ?? ""); ?>">
                                            <div class="small">subir logo manualmente y se completará la url automaticamente al actualizar el canal:</div>
                                            <input type="file" name="channel_icon_file" accept="image/png, image/jpeg, image/gif, image/webp">
                                        </div>
                                        <div class="label-input"><label>Tipo:</label><br>
                                            <select name="channel_type" onchange="toggleDrmFieldEdit('<?php echo $sectionIndex; ?>','<?php echo $channelIndex; ?>','visible', this.value)">
                                                <option value="HLS" <?php echo (($channel['type'] ?? 'HLS') === 'HLS' ? 'selected' : ''); ?>>HLS</option>
                                                <option value="WEBVIEW" <?php echo (($channel['type'] ?? '') === 'WEBVIEW' ? 'selected' : ''); ?>>WEBVIEW</option>
                                                <option value="CLEARKEY" <?php echo (($channel['type'] ?? '') === 'CLEARKEY' ? 'selected' : ''); ?>>DRM (CLEARKEY)</option>
                                            </select>
                                        </div>
                                        <?php $displayDrm = (($channel['type'] ?? '') === 'CLEARKEY') ? 'block' : 'none'; $drmVal = isset($channel['drm_license_uri']) ? decrypt_data($channel['drm_license_uri']) : ''; ?>
                                        <div class="label-input" id="drmFieldEdit_visible_<?php echo $sectionIndex; ?>_<?php echo $channelIndex; ?>" style="display: <?php echo $displayDrm; ?>;"><label>URL de la licencia DRM:</label><br><input type="text" name="channel_drm" class="input-text" value="<?php echo htmlspecialchars($drmVal); ?>"></div>
                                        <div class="label-input"><label>Nueva posición:</label><br><input type="number" name="channel_new_position" value="<?php echo $channelIndex+1; ?>" style="width:80px;"></div>
                                        <?php $headersChecked = isset($channel['headers']) && is_array($channel['headers']); ?>
                                        <div><label>¿Headers?</label> <input type="radio" name="has_headers" value="yes" <?php echo $headersChecked ? 'checked' : ''; ?> onclick="document.getElementById('headersEdit_visible_<?php echo $sectionIndex; ?>_<?php echo $channelIndex; ?>').style.display='block';"> Sí <input type="radio" name="has_headers" value="no" <?php echo !$headersChecked ? 'checked' : ''; ?> onclick="document.getElementById('headersEdit_visible_<?php echo $sectionIndex; ?>_<?php echo $channelIndex; ?>').style.display='none';"> No</div>
                                        <div id="headersEdit_visible_<?php echo $sectionIndex; ?>_<?php echo $channelIndex; ?>" style="display: <?php echo $headersChecked?'block':'none'; ?>;"><p>Headers:</p><div id="headerPairsEdit_visible_<?php echo $sectionIndex; ?>_<?php echo $channelIndex; ?>">
                                            <?php if ($headersChecked) { foreach ($channel['headers'] as $hk => $hv) { echo '<div class="header-pair"><input type="text" name="channel_header_key[]" value="'.htmlspecialchars($hk).'" placeholder="Header..." /><input type="text" name="channel_header_val[]" value="'.htmlspecialchars(decrypt_data($hv)).'" placeholder="Valor..." /></div>'; } } else { echo '<div class="header-pair"><input type="text" name="channel_header_key[]" placeholder="Header..." /><input type="text" name="channel_header_val[]" placeholder="Valor..." /></div>'; } ?>
                                        </div><button type="button" class="add-header-btn btn-secondary" onclick="addHeaderPair('headerPairsEdit_visible_<?php echo $sectionIndex; ?>_<?php echo $channelIndex; ?>')">Agregar otro header</button></div>
                                        <div style="margin-top: 10px;"><button type="submit" class="btn-primary">Actualizar canal</button></div>
                                    </form>
                                </div>
                            </li>
                        <?php endforeach; ?>

                        <?php if (!empty($hiddenChannels)): ?>
                            <li class="hidden-divider"><em>Canales ocultos</em></li>
                            <?php foreach ($hiddenChannels as $channelIndex => $channel): ?>
                                <li class="channel-hidden">
                                    <strong><?php echo htmlspecialchars($channel['name']); ?></strong> <span class="badge-oculto">Oculto</span>
                                    <form method="POST" class="inline-form">
                                        <input type="hidden" name="action" value="delete_channel"><input type="hidden" name="section_index" value="<?php echo $sectionIndex; ?>"><input type="hidden" name="channel_index" value="<?php echo $channelIndex; ?>"><input type="hidden" name="channel_source" value="hidden">
                                        <button type="submit" class="btn-danger" onclick="return confirm('¿Eliminar este canal oculto?')">Eliminar</button>
                                    </form>
                                    <form method="POST" class="inline-form">
                                        <input type="hidden" name="action" value="unhide_channel"><input type="hidden" name="section_index" value="<?php echo $sectionIndex; ?>"><input type="hidden" name="channel_index" value="<?php echo $channelIndex; ?>">
                                        <button type="submit" class="btn-primary">Habilitar</button>
                                    </form>
                                    <button type="button" class="btn-secondary" onclick="toggleEditForm('<?php echo $sectionIndex; ?>','<?php echo $channelIndex; ?>','hidden')">Editar</button>

                                    <div id="editForm_hidden_<?php echo $sectionIndex; ?>_<?php echo $channelIndex; ?>" class="edit-channel-form" style="display:none;">
                                        <h4>Editar canal oculto</h4>
                                        <form method="POST" enctype="multipart/form-data">
                                            <input type="hidden" name="action" value="edit_channel"><input type="hidden" name="section_index" value="<?php echo $sectionIndex; ?>"><input type="hidden" name="channel_index" value="<?php echo $channelIndex; ?>"><input type="hidden" name="channel_source" value="hidden">
                                            <div class="label-input"><label>Nombre:</label><br><input type="text" name="channel_name" class="input-text" value="<?php echo htmlspecialchars($channel['name']); ?>" required></div>
                                            <div class="label-input"><label>URL del canal:</label><br><input type="text" name="channel_url" class="input-text" value="<?php echo htmlspecialchars(decrypt_data($channel['url'])); ?>"></div>
                                            <div class="label-input">
                                                <label>URL del icono:</label><br>
                                                <input type="text" name="channel_icon" class="input-text" value="<?php echo htmlspecialchars($channel['icono'] ?? ""); ?>">
                                                <div class="small">subir logo manualmente y se completará la url automaticamente al actualizar el canal:</div>
                                                <input type="file" name="channel_icon_file" accept="image/png, image/jpeg, image/gif, image/webp">
                                            </div>
                                            <div class="label-input"><label>Tipo:</label><br>
                                                <select name="channel_type" onchange="toggleDrmFieldEdit('<?php echo $sectionIndex; ?>','<?php echo $channelIndex; ?>','hidden', this.value)">
                                                    <option value="HLS" <?php echo (($channel['type'] ?? 'HLS') === 'HLS' ? 'selected' : ''); ?>>HLS</option>
                                                    <option value="WEBVIEW" <?php echo (($channel['type'] ?? '') === 'WEBVIEW' ? 'selected' : ''); ?>>WEBVIEW</option>
                                                    <option value="CLEARKEY" <?php echo (($channel['type'] ?? '') === 'CLEARKEY' ? 'selected' : ''); ?>>DRM (CLEARKEY)</option>
                                                </select>
                                            </div>
                                            <?php $displayDrmHidden = (($channel['type'] ?? '') === 'CLEARKEY') ? 'block' : 'none'; $drmValHidden = isset($channel['drm_license_uri']) ? decrypt_data($channel['drm_license_uri']) : ''; ?>
                                            <div class="label-input" id="drmFieldEdit_hidden_<?php echo $sectionIndex; ?>_<?php echo $channelIndex; ?>" style="display: <?php echo $displayDrmHidden; ?>;"><label>URL de la licencia DRM:</label><br><input type="text" name="channel_drm" class="input-text" value="<?php echo htmlspecialchars($drmValHidden); ?>"></div>
                                            <?php $headersHiddenChecked = isset($channel['headers']) && is_array($channel['headers']); ?>
                                            <div><label>¿Headers?</label> <input type="radio" name="has_headers" value="yes" <?php echo $headersHiddenChecked ? 'checked' : ''; ?> onclick="document.getElementById('headersEdit_hidden_<?php echo $sectionIndex; ?>_<?php echo $channelIndex; ?>').style.display='block';"> Sí <input type="radio" name="has_headers" value="no" <?php echo !$headersHiddenChecked ? 'checked' : ''; ?> onclick="document.getElementById('headersEdit_hidden_<?php echo $sectionIndex; ?>_<?php echo $channelIndex; ?>').style.display='none';"> No</div>
                                            <div id="headersEdit_hidden_<?php echo $sectionIndex; ?>_<?php echo $channelIndex; ?>" style="display: <?php echo $headersHiddenChecked?'block':'none'; ?>;"><p>Headers:</p><div id="headerPairsEdit_hidden_<?php echo $sectionIndex; ?>_<?php echo $channelIndex; ?>">
                                                <?php if ($headersHiddenChecked) { foreach ($channel['headers'] as $hk => $hv) { echo '<div class="header-pair"><input type="text" name="channel_header_key[]" value="'.htmlspecialchars($hk).'" placeholder="Header..." /><input type="text" name="channel_header_val[]" value="'.htmlspecialchars(decrypt_data($hv)).'" placeholder="Valor..." /></div>'; } } else { echo '<div class="header-pair"><input type="text" name="channel_header_key[]" placeholder="Header..." /><input type="text" name="channel_header_val[]" placeholder="Valor..." /></div>'; } ?>
                                            </div><button type="button" class="add-header-btn btn-secondary" onclick="addHeaderPair('headerPairsEdit_hidden_<?php echo $sectionIndex; ?>_<?php echo $channelIndex; ?>')">Agregar otro header</button></div>
                                            <p class="note-hidden-edit">Para cambiar la posición, habilita el canal primero.</p>
                                            <div style="margin-top: 10px;"><button type="submit" class="btn-primary">Guardar cambios</button></div>
                                        </form>
                                    </div>
                                </li>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    <?php else: ?>
                        <li><em>No hay canales en esta sección.</em></li>
                    <?php endif; ?>
                </ul>

                <div class="add-channel-form">
                    <h4>Agregar nuevo canal a "<?php echo htmlspecialchars($section['name']); ?>"</h4>
                    <form method="POST" enctype="multipart/form-data">
                        <input type="hidden" name="action" value="add_channel"><input type="hidden" name="section_index" value="<?php echo $sectionIndex; ?>">
                        <div class="label-input"><label>Nombre:</label><br><input type="text" name="channel_name" class="input-text" required></div>
                        <div class="label-input"><label>URL del canal:</label><br><input type="text" name="channel_url" class="input-text"></div>
                        <div class="label-input">
                            <label>URL del icono:</label><br>
                            <input type="text" name="channel_icon" class="input-text">
                            <div class="small">subir logo manualmente y se completará la url automaticamente al agregar el canal:</div>
                            <input type="file" name="channel_icon_file" accept="image/png, image/jpeg, image/gif, image/webp">
                        </div>
                        <div class="label-input"><label>Tipo:</label><br>
                            <select name="channel_type" onchange="toggleDrmField(this, '<?php echo $sectionIndex; ?>')">
                                <option value="HLS">HLS</option>
                                <option value="WEBVIEW">WEBVIEW</option>
                                <option value="CLEARKEY">DRM (CLEARKEY)</option>
                            </select>
                        </div>
                        <div class="label-input" id="drmField_<?php echo $sectionIndex; ?>" style="display:none;"><label>URL de la licencia DRM:</label><br><input type="text" name="channel_drm" class="input-text"></div>
                        <div><label>¿Headers?</label> <input type="radio" name="has_headers" value="yes" onclick="document.getElementById('headersContainer_<?php echo $sectionIndex; ?>').style.display='block';"> Sí <input type="radio" name="has_headers" value="no" checked onclick="document.getElementById('headersContainer_<?php echo $sectionIndex; ?>').style.display='none';"> No</div>
                        <div id="headersContainer_<?php echo $sectionIndex; ?>" style="display:none;"><p>Headers:</p><div id="headerPairs_<?php echo $sectionIndex; ?>"><div class="header-pair"><input type="text" name="channel_header_key[]" placeholder="Header..." /><input type="text" name="channel_header_val[]" placeholder="Valor..." /></div></div><button type="button" class="add-header-btn btn-secondary" onclick="addHeaderPair('headerPairs_<?php echo $sectionIndex; ?>')">Agregar otro header</button></div>
                        <div style="margin-top: 10px;"><button type="submit" class="btn-primary">Agregar canal</button></div>
                    </form>
                </div>
            </div>
        <?php endforeach; ?>

        <?php if (!empty($hiddenSections)): ?>
            <div class="section-hidden-box">
                <h3>Secciones ocultas</h3>
                <ul class="channels-list">
                    <?php foreach ($hiddenSections as $sectionIndex => $section): ?>
                        <li class="channel-hidden">
                            <strong><?php echo htmlspecialchars($section['name']); ?></strong> <span class="badge-oculto">Oculta</span>
                            <form method="POST" class="inline-form" style="margin-left:10px;">
                                <input type="hidden" name="action" value="show_section"><input type="hidden" name="section_index" value="<?php echo $sectionIndex; ?>">
                                <button type="submit" class="btn-secondary">Mostrar</button>
                            </form>
                            <form method="POST" class="inline-form">
                                <input type="hidden" name="action" value="delete_section"><input type="hidden" name="section_index" value="<?php echo $sectionIndex; ?>">
                                <button type="submit" class="btn-danger" onclick="return confirm('¿Eliminar esta sección oculta?');">Eliminar</button>
                            </form>
                        </li>
                    <?php endforeach; ?>
                </ul>
            </div>
        <?php endif; ?>
    </div>

    <!-- Contenedor del Editor JSON -->
    <div id="jsonEditorContainer" class="panel-section">
        <div class="top-controls">
            <h1>Editor de JSON Normal</h1>
            <button id="hideJsonEditorBtn" class="btn-secondary">Volver al Panel Normal</button>
        </div>
        <div class="json-editor-warning">
            ¡CUIDADO! Estás editando el archivo JSON directamente. Un error de sintaxis puede hacer que la aplicación deje de funcionar.
        </div>
        
        <?php if (!empty($json_editor_error)): ?>
            <div id="syntaxError"><?= htmlspecialchars($json_editor_error) ?></div>
        <?php endif; ?>

        <div id="findReplaceBar">
            <input type="text" id="findInput" placeholder="Buscar..."><input type="text" id="replaceInput" placeholder="Reemplazar con...">
            <button id="replaceBtn" class="btn-warning">Reemplazar</button><button id="replaceAllBtn" class="btn-warning">Reemplazar Todo</button>
            <button id="closeFindBtn" class="btn-danger" style="float:right;">X</button>
        </div>

        <form method="POST">
            <input type="hidden" name="action" value="save_raw_json">
            <textarea name="json_content" id="json_content_editor"><?php echo htmlspecialchars($json_for_editor); ?></textarea>
            <div style="text-align: right; margin-top: 15px;"><button type="submit" class="btn-primary" style="padding: 10px 20px;">Guardar Cambios</button></div>
        </form>
    </div>

</div>

<!-- HTML para la Modal de Actualización de Bearer -->
<div id="bearerModal" class="modal-overlay">
  <div class="modal-content">
    <h3>Actualizar Bearer Token (local)</h3>
    <p>Pegá el nuevo Bearer token completo. Se validará que comience con "Bearer " y que no tenga espacios extra. Se guardará en <strong>bearer.json</strong>.</p>
    <form method="POST">
        <input type="hidden" name="action" value="update_bearer">
        <textarea name="new_bearer_token" placeholder="Pega el nuevo Bearer aquí, por ejemplo: Bearer eyJ0eXAiOi..."></textarea>
        <div class="modal-buttons">
            <button type="button" id="closeBearerModalBtn" class="btn-secondary">Cancelar</button>
            <button type="submit" class="btn-primary">Guardar Bearer</button>
        </div>
    </form>
  </div>
</div>

<script>
// Funciones del panel original
function toggleEditForm(s, c, src) { src = src || 'visible'; var e = document.getElementById('editForm_'+src+'_'+s+'_'+c); if (!e) return; var current = window.getComputedStyle(e).display; e.style.display = (current === 'none') ? 'block' : 'none'; }
function toggleRenameForm(s) { var e = document.getElementById('renameForm_'+s); e.style.display = (e.style.display === 'none') ? 'block' : 'none'; }
function toggleDrmField(el, s) { document.getElementById('drmField_'+s).style.display = (el.value === 'CLEARKEY') ? 'block' : 'none'; }
function toggleDrmFieldEdit(s, c, src, v) { src = src || 'visible'; var d = document.getElementById('drmFieldEdit_'+src+'_'+s+'_'+c); if (!d) return; d.style.display = (v === 'CLEARKEY') ? 'block' : 'none'; }
function addHeaderPair(id) {
    var c = document.getElementById(id); if (!c) return;
    var d = document.createElement('div'); d.className = 'header-pair';
    d.innerHTML = '<input type="text" name="channel_header_key[]" placeholder="Header..." /> <input type="text" name="channel_header_val[]" placeholder="Valor..." />';
    c.appendChild(d);
}

// Lógica para el editor JSON y la modal del Bearer
document.addEventListener('DOMContentLoaded', function() {
    // NUEVO: Habilitar arrastrar y soltar sin dependencias externas para los banners
    const bannerList = document.getElementById('banner-list');
    const reorderForm = document.getElementById('reorder-banners-form');
    const orderInputsContainer = document.getElementById('banner-order-inputs');

    if (bannerList && reorderForm && orderInputsContainer) {
        let draggedItem = null;

        const updateBannerOrderInputs = () => {
            orderInputsContainer.innerHTML = '';
            const urls = Array.from(bannerList.querySelectorAll('.banner-item'))
                .map(item => item.getAttribute('data-url'))
                .filter(Boolean);
            urls.forEach(url => {
                const input = document.createElement('input');
                input.type = 'hidden';
                input.name = 'banner_order[]';
                input.value = url;
                orderInputsContainer.appendChild(input);
            });
        };

        bannerList.querySelectorAll('.banner-item').forEach(item => {
            item.setAttribute('draggable', 'true');
        });

        bannerList.addEventListener('dragstart', event => {
            const target = event.target.closest('.banner-item');
            if (!target) return;
            draggedItem = target;
            event.dataTransfer.effectAllowed = 'move';
            try {
                event.dataTransfer.setData('text/plain', '');
            } catch (err) {
                // Algunos navegadores pueden lanzar una excepción si no se proporciona dato
            }
            draggedItem.classList.add('dragging');
        });

        bannerList.addEventListener('dragend', () => {
            if (!draggedItem) return;
            draggedItem.classList.remove('dragging');
            draggedItem = null;
            updateBannerOrderInputs();
        });

        bannerList.addEventListener('dragover', event => {
            if (!draggedItem) return;
            event.preventDefault();
            const target = event.target.closest('.banner-item');

            if (!target || target === draggedItem) {
                if (!target && event.target === bannerList) {
                    bannerList.appendChild(draggedItem);
                }
                return;
            }

            const targetRect = target.getBoundingClientRect();
            const isAfter = (event.clientY - targetRect.top) > (targetRect.height / 2);
            if (isAfter) {
                target.after(draggedItem);
            } else {
                target.before(draggedItem);
            }
        });

        bannerList.addEventListener('drop', event => {
            event.preventDefault();
            updateBannerOrderInputs();
        });

        reorderForm.addEventListener('submit', () => {
            updateBannerOrderInputs();
        });

        updateBannerOrderInputs();
    }

    const panelNormal = document.getElementById('panelNormal');
    const jsonEditorContainer = document.getElementById('jsonEditorContainer');
    const showJsonEditorBtn = document.getElementById('showJsonEditorBtn');
    const hideJsonEditorBtn = document.getElementById('hideJsonEditorBtn');

    if(showJsonEditorBtn) {
        showJsonEditorBtn.addEventListener('click', () => {
            panelNormal.style.display = 'none';
            jsonEditorContainer.style.display = 'block';
            window.history.pushState(null, '', '?view=editor');
        });
    }
    if(hideJsonEditorBtn) {
        hideJsonEditorBtn.addEventListener('click', () => {
            panelNormal.style.display = 'block';
            jsonEditorContainer.style.display = 'none';
            window.history.pushState(null, '', window.location.pathname);
        });
    }
    if (new URLSearchParams(window.location.search).get('view') === 'editor') {
        if(showJsonEditorBtn) showJsonEditorBtn.click();
    }

    // Lógica para la modal del Bearer
    const bearerModal = document.getElementById('bearerModal');
    const updateBearerBtn = document.getElementById('updateBearerBtn');
    const closeBearerModalBtn = document.getElementById('closeBearerModalBtn');
    if(updateBearerBtn) updateBearerBtn.addEventListener('click', () => { bearerModal.style.display = 'block'; });
    if(closeBearerModalBtn) closeBearerModalBtn.addEventListener('click', () => { bearerModal.style.display = 'none'; });
    window.addEventListener('click', (event) => { if (event.target == bearerModal) bearerModal.style.display = 'none'; });

    // Lógica de búsqueda y reemplazo
    const editor = document.getElementById('json_content_editor');
    const findReplaceBar = document.getElementById('findReplaceBar');
    const findInput = document.getElementById('findInput');
    const replaceInput = document.getElementById('replaceInput');
    const replaceBtn = document.getElementById('replaceBtn');
    const replaceAllBtn = document.getElementById('replaceAllBtn');
    const closeFindBtn = document.getElementById('closeFindBtn');
    if (editor) {
        editor.addEventListener('keydown', (e) => {
            if (e.ctrlKey && e.key === 'f') { e.preventDefault(); findReplaceBar.style.display = 'block'; findInput.focus(); }
        });
    }
    if (closeFindBtn) closeFindBtn.addEventListener('click', () => { findReplaceBar.style.display = 'none'; });
    if (replaceBtn) {
        replaceBtn.addEventListener('click', () => {
            const findValue = findInput.value; const replaceValue = replaceInput.value;
            if (findValue) editor.value = editor.value.replace(findValue, replaceValue);
        });
    }
    if(replaceAllBtn) {
        replaceAllBtn.addEventListener('click', () => {
            const findValue = findInput.value; const replaceValue = replaceInput.value;
            if (findValue) {
                const regex = new RegExp(findValue.replace(/[-\/\\^$*+?.()|[\]{}]/g, '\\$&'), 'g');
                editor.value = editor.value.replace(regex, replaceValue);
            }
        });
    }
});
</script>
</body>
</html>
